@php
    // Normalize data for Cart vs Landing Page
    $items = $cart ?? [];
    $subtotal = $total ?? 0;
    $isLanding = isset($landingPage) && $landingPage->product;
    
    if ($isLanding && empty($items)) {
        $items = [
            [
                'name' => $landingPage->product->name,
                'thumbnail' => $landingPage->product->thumbnail,
                'price' => $landingPage->product->price,
                'quantity' => 1,
                'color_name' => null,
                'size_name' => null,
            ]
        ];
        $subtotal = $landingPage->product->price;
    }
@endphp

@if (session('error'))
    <div class="mb-6 bg-red-50 border-l-4 border-red-500 text-red-700 px-6 py-4 rounded-r-lg shadow-sm" role="alert">
        <span class="block text-sm font-bold">{{ session('error') }}</span>
    </div>
@endif

@if ($errors->any())
    <div class="mb-6 bg-red-50 border-l-4 border-red-500 text-red-700 px-6 py-4 rounded-r-lg shadow-sm" role="alert">
        <strong class="font-bold text-sm block mb-2">অনুগ্রহ করে নিচের ভুলগুলো সংশোধন করুন:</strong>
        <ul class="list-disc list-inside text-sm space-y-1">
            @foreach ($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
    </div>
@endif

{{-- Hidden fields for Landing Page --}}
@php
    $productId = old('product_id') ?? request('product_id');
    $quantity = old('quantity') ?? request('quantity', 1);
    $colorId = old('color_id') ?? request('color_id');
    $sizeId = old('size_id') ?? request('size_id');
@endphp

@if(isset($landingPage))
    <input type="hidden" name="source" value="landing_page">
    @if($landingPage->product)
        <input type="hidden" name="product_id" value="{{ $landingPage->product->id }}">
        <input type="hidden" name="quantity" value="1">
    @endif
@elseif($productId)
    <input type="hidden" name="product_id" value="{{ $productId }}">
    <input type="hidden" name="quantity" value="{{ $quantity }}">
    @if($colorId)
        <input type="hidden" name="color_id" value="{{ $colorId }}">
    @endif
    @if($sizeId)
        <input type="hidden" name="size_id" value="{{ $sizeId }}">
    @endif
@endif

<input type="hidden" name="incomplete_order_id" id="incomplete_order_id" value="{{ session('incomplete_order_id') }}">

<div class="grid grid-cols-1 lg:grid-cols-12 gap-8 items-start">
    
    <!-- LEFT COLUMN: Forms -->
    <div class="lg:col-span-7 space-y-6">
        
        <!-- Contact Information -->
        <div class="bg-white p-6 lg:p-8 rounded-xl shadow-sm border border-gray-200">
            <h2 class="text-lg font-bold text-gray-900 mb-6 uppercase tracking-wide flex items-center gap-2">
                <span class="text-theme-primary">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path></svg>
                </span>
                যোগাযোগ তথ্য
            </h2>
            <div class="space-y-4">
                {{-- Only show email field if it's not a landing page or if you want to collect it there too. 
                     The landing page example didn't have email, but the main checkout does. 
                     We'll keep it as per "exact design". --}}
                <div>
                    <label for="email" class="block text-sm font-bold text-gray-700 mb-1">ইমেইল অ্যাড্রেস <span class="text-gray-400 font-normal">(অপশনাল)</span></label>
                    <input type="email" name="email" id="email" value="{{ old('email') }}" 
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
                <div>
                    <label for="phone" class="block text-sm font-bold text-gray-700 mb-1">ফোন নম্বর <span class="text-red-500">*</span></label>
                    <input type="text" name="phone" id="phone" value="{{ old('phone') }}" required placeholder="017XXXXXXXX"
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
            </div>
        </div>

        <!-- Shipping Address -->
        <div class="bg-white p-6 lg:p-8 rounded-xl shadow-sm border border-gray-200">
            <h2 class="text-lg font-bold text-gray-900 mb-6 uppercase tracking-wide flex items-center gap-2">
                <span class="text-theme-primary">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                </span>
                শিপিং ঠিকানা
            </h2>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="first_name" class="block text-sm font-bold text-gray-700 mb-1">আপনার নাম <span class="text-gray-400 font-normal">(অপশনাল)</span></label>
                    <input type="text" name="first_name" id="first_name" value="{{ old('first_name') }}" 
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
                <div>
                    <label for="last_name" class="block text-sm font-bold text-gray-700 mb-1">নামের শেষ অংশ <span class="text-gray-400 font-normal">(অপশনাল)</span></label>
                    <input type="text" name="last_name" id="last_name" value="{{ old('last_name') }}" 
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
                <div class="md:col-span-2">
                    <label for="address" class="block text-sm font-bold text-gray-700 mb-1">ঠিকানা <span class="text-red-500">*</span></label>
                    <input type="text" name="address" id="address" value="{{ old('address') }}" required placeholder="রাস্তা / বাসা নম্বর / এলাকা"
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
                <div class="md:col-span-2">
                    <label class="block text-sm font-bold text-gray-700 mb-2">শিপিং এলাকা <span class="text-red-500">*</span></label>
                    <div class="grid grid-cols-2 gap-4">
                        @if(isset($shippingCharges))
                            @foreach($shippingCharges as $charge)
                            <label class="relative flex items-center justify-between p-4 rounded-lg border border-gray-300 cursor-pointer hover:border-theme-primary hover:bg-orange-50 transition-colors group">
                                <span class="flex items-center">
                                    <input type="radio" name="area_id" value="{{ $charge->id }}" required class="h-4 w-4 text-theme-primary border-gray-300 focus:ring-theme-primary">
                                    <span class="ml-3 block text-sm font-bold text-gray-900">{{ $charge->area_name }}</span>
                                </span>
                            </label>
                            @endforeach
                        @endif
                    </div>
                </div>
                <div>
                    <label for="city" class="block text-sm font-bold text-gray-700 mb-1">শহর <span class="text-gray-400 font-normal">(অপশনাল)</span></label>
                    <input type="text" name="city" id="city" value="{{ old('city') }}" 
                        class="block w-full rounded-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-500 transition-all shadow-sm">
                </div>
                <div>
                    <label for="postal_code" class="block text-sm font-bold text-gray-700 mb-1">পোস্ট কোড <span class="text-gray-400 font-normal">(অপশনাল)</span></label>
                    <input type="text" name="postal_code" id="postal_code" value="{{ old('postal_code') }}" 
                        class="block w-full rounded-lg border-gray-300 focus:border-theme-primary focus:ring-theme-primary py-3 px-4 text-gray-900 placeholder-gray-400 transition-shadow">
                </div>
            </div>
        </div>

        <!-- Payment Method -->
        <div class="bg-white p-6 lg:p-8 rounded-xl shadow-sm border border-gray-200">
             <h2 class="text-lg font-bold text-gray-900 mb-6 uppercase tracking-wide flex items-center gap-2">
                <span class="text-theme-primary">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"></path></svg>
                </span>
                পেমেন্ট মেথড
            </h2>
            <div class="space-y-3">
                <label class="relative flex items-center p-4 rounded-lg border border-theme-primary-light bg-theme-primary-light cursor-pointer hover:border-theme-primary transition-colors group">
                    <input id="cod" name="payment_method" type="radio" value="cod" checked class="h-4 w-4 text-theme-primary border-gray-300 focus:ring-theme-primary">
                    <div class="ml-4">
                        <span class="block text-sm font-bold text-gray-900">ক্যাশ অন ডেলিভারি</span>
                        <span class="block text-xs text-gray-500 mt-0.5">পণ্য হাতে পেয়ে মূল্য পরিশোধ করুন</span>
                    </div>
                    <div class="ml-auto">
                        <svg class="w-5 h-5 text-theme-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>
                    </div>
                </label>

                @if(isset($settings) && $settings->bkash_enabled)
                <label class="relative flex items-center p-4 rounded-lg border border-gray-200 cursor-pointer hover:border-pink-500 hover:bg-pink-50 transition-colors group">
                    <input id="bkash" name="payment_method" type="radio" value="bkash" class="h-4 w-4 text-pink-600 border-gray-300 focus:ring-pink-500">
                    <div class="ml-4 flex-1">
                        <div class="flex items-center gap-2">
                            <span class="block text-sm font-bold text-gray-900">বিকাশ</span>
                            @if($settings->bkash_test_mode)
                                <span class="text-[10px] bg-pink-100 text-pink-600 px-2 py-0.5 rounded-full font-bold uppercase">Test</span>
                            @endif
                        </div>
                    </div>
                </label>
                @endif

                @if(isset($settings) && $settings->sslcz_enabled)
                <label class="relative flex items-center p-4 rounded-lg border border-gray-300 cursor-pointer hover:border-blue-500 hover:bg-blue-50 transition-colors group">
                    <input id="sslcommerz" name="payment_method" type="radio" value="sslcommerz" class="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                    <div class="ml-4 flex-1">
                        <div class="flex items-center gap-2">
                            <span class="block text-sm font-bold text-gray-900">এসএসএল কমার্স</span>
                            @if($settings->sslcz_test_mode)
                                <span class="text-[10px] bg-blue-100 text-blue-600 px-2 py-0.5 rounded-full font-bold uppercase">Test</span>
                            @endif
                        </div>
                    </div>
                </label>
                @endif
            </div>
        </div>

        <!-- Submit Button (Mobile Order: shown at bottom) -->
        <button type="submit" class="w-full bg-gray-900 text-white py-5 rounded-xl font-bold uppercase tracking-wider hover:bg-theme-primary transition-colors duration-300 shadow-lg text-sm flex items-center justify-center gap-2">
            <span>অর্ডার কনফার্ম করুন</span>
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14 5l7 7m0 0l-7 7m7-7H3"></path></svg>
        </button>

    </div>

    <!-- RIGHT COLUMN: Summary -->
    <div class="lg:col-span-5 space-y-6">
        <div class="bg-white p-6 lg:p-8 rounded-xl shadow-sm border border-gray-200 sticky top-24">
            <h2 class="text-lg font-bold text-gray-900 mb-6 uppercase tracking-wide">অর্ডার সামারি</h2>
            
            <!-- Totals -->
            <div class="space-y-3 mb-6 pb-6 border-b border-gray-100">
                <div class="flex justify-between text-gray-600 text-sm">
                    <span>সাবটোটাল</span>
                    <span class="font-bold text-gray-900">৳{{ number_format($subtotal, 0) }}</span>
                </div>
                <div class="flex justify-between text-gray-600 text-sm">
                    <span>শিপিং চার্জ</span>
                    <span id="shipping_cost_display" class="font-bold text-gray-900">৳0</span>
                </div>
            </div>

            <div class="flex justify-between items-end mb-8">
                <span class="text-base font-bold text-gray-900">সর্বমোট</span>
                <span id="total_amount_display" class="text-2xl font-heading font-bold text-theme-primary" data-subtotal="{{ $subtotal }}">৳{{ number_format($subtotal, 0) }}</span>
            </div>

            <!-- PROMO CODE SECTION -->
            <div class="mb-8">
                <label class="block text-sm font-bold text-gray-700 mb-2">প্রোমো কোড বা ডিসকাউন্ট</label>
                <div class="flex">
                    <input type="text" placeholder="কুপন কোড দিন" class="flex-1 rounded-l-lg border-gray-400 bg-gray-50 focus:bg-white focus:border-theme-primary focus:ring-theme-primary text-sm py-2.5 px-3">
                    <button type="button" class="bg-gray-900 text-white px-6 py-2.5 rounded-r-lg font-bold text-sm hover:bg-theme-primary transition-colors uppercase tracking-wide">
                        এপ্লাই
                    </button>
                </div>
            </div>

            <!-- BAG SUMMARY -->
            <div class="pt-6 border-t border-gray-100">
                <h3 class="text-sm font-bold text-gray-900 mb-4 flex items-center justify-between">
                    <span>ব্যাগে থাকা পণ্য</span>
                    <span class="bg-gray-900 text-white text-xs px-2 py-0.5 rounded-full">{{ count($items) }}</span>
                </h3>
                
                <div class="space-y-4 max-h-[400px] overflow-y-auto pr-2 custom-scrollbar">
                    @foreach($items as $item)
                        <div class="flex gap-4 items-start">
                            <div class="w-16 h-16 rounded-lg border border-gray-100 bg-gray-50 overflow-hidden flex-shrink-0 relative">
                                @php
                                    // Handle image source for both Storage URL and direct asset/URL
                                    $imgSrc = '';
                                    if(isset($item['thumbnail'])) {
                                        if(str_starts_with($item['thumbnail'], 'http') || str_starts_with($item['thumbnail'], '/')) {
                                             $imgSrc = asset($item['thumbnail']);
                                        } else {
                                             $imgSrc = Storage::url($item['thumbnail']);
                                        }
                                    }
                                @endphp

                                @if($imgSrc)
                                    <img src="{{ $imgSrc }}" alt="{{ $item['name'] }}" class="w-full h-full object-cover">
                                @else
                                    <div class="w-full h-full flex items-center justify-center text-gray-300">
                                        <i class="fas fa-image"></i>
                                    </div>
                                @endif
                                <span class="absolute top-0 right-0 bg-theme-primary text-white text-[10px] font-bold px-1.5 py-0.5 rounded-bl">x{{ $item['quantity'] }}</span>
                            </div>
                            <div class="flex-1 min-w-0">
                                <h4 class="text-sm font-medium text-gray-900 line-clamp-2 mb-1">{{ $item['name'] }}</h4>
                                @if(isset($item['color_name']) || isset($item['size_name']))
                                    <p class="text-xs text-gray-500 mb-1">
                                        {{ $item['color_name'] ?? '' }} {{ isset($item['color_name']) && isset($item['size_name']) ? '/' : '' }} {{ $item['size_name'] ?? '' }}
                                    </p>
                                @endif
                                <p class="text-sm font-bold text-theme-primary">৳{{ number_format($item['price'] * $item['quantity'], 0) }}</p>
                            </div>
                        </div>
                    @endforeach
                </div>
            </div>

        </div>
    </div>

</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const form = document.querySelector('form[action="{{ route('checkout.store') }}"]');
        const phoneInput = document.getElementById('phone');
        let timeoutId;

        if (form && phoneInput) {
            function saveIncompleteOrder() {
                const formData = new FormData(form);
                const phone = formData.get('phone');

                // Only save if phone number is present
                if (!phone || phone.trim() === '') return;

                fetch('{{ route('checkout.storeIncomplete') }}', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    },
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.status === 'success' && data.order_id) {
                        const hiddenInput = document.getElementById('incomplete_order_id');
                        if (hiddenInput) {
                            hiddenInput.value = data.order_id;
                        }
                    }
                })
                .catch(error => {
                    console.log('Incomplete order save ignored/failed', error);
                });
            }

            // Debounce function
            function debounce(func, wait) {
                return function() {
                    const context = this, args = arguments;
                    clearTimeout(timeoutId);
                    timeoutId = setTimeout(() => func.apply(context, args), wait);
                };
            }

            phoneInput.addEventListener('input', debounce(saveIncompleteOrder, 1000));
            phoneInput.addEventListener('blur', saveIncompleteOrder);
        }

        // --- SHIPPING COST CALCULATION ---
        const areaInputs = document.querySelectorAll('input[name="area_id"]');
        const shippingCostDisplay = document.getElementById('shipping_cost_display');
        const totalAmountDisplay = document.getElementById('total_amount_display');
        const baseSubtotal = parseFloat(totalAmountDisplay.dataset.subtotal);
        
        function formatCurrency(amount) {
            return '৳' + new Intl.NumberFormat('en-BD').format(amount);
        }

        function handleShippingChange() {
            const areaId = this.value;
            if (!areaId) {
                shippingCostDisplay.textContent = '৳0';
                totalAmountDisplay.textContent = formatCurrency(baseSubtotal);
                return;
            }

            const formData = new FormData();
            formData.append('area_id', areaId);
            formData.append('_token', '{{ csrf_token() }}');
            
            // Add other parameters if they exist (for direct checkout calculation)
            const productIdInput = document.querySelector('input[name="product_id"]');
            const quantityInput = document.querySelector('input[name="quantity"]');
            const colorIdInput = document.querySelector('input[name="color_id"]');
            const sizeIdInput = document.querySelector('input[name="size_id"]');

            if (productIdInput) formData.append('product_id', productIdInput.value);
            if (quantityInput) formData.append('quantity', quantityInput.value);
            if (colorIdInput) formData.append('color_id', colorIdInput.value);
            if (sizeIdInput) formData.append('size_id', sizeIdInput.value);

            shippingCostDisplay.textContent = '...';

            fetch('{{ route("checkout.calculate_shipping") }}', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.cost !== undefined) {
                    shippingCostDisplay.textContent = data.formatted_cost;
                    totalAmountDisplay.textContent = data.formatted_total;
                }
            })
            .catch(error => {
                console.error('Error calculating shipping:', error);
                shippingCostDisplay.textContent = 'Error';
            });
        }

        areaInputs.forEach(input => {
            input.addEventListener('change', handleShippingChange);
        });
        
        // Trigger calculation if area is already selected (e.g. old input)
        const selectedArea = document.querySelector('input[name="area_id"]:checked');
        if (selectedArea) {
            handleShippingChange.call(selectedArea);
        }
    });
</script>