<x-admin-layout>
    <div class="py-6 px-4 sm:px-6 lg:px-8 bg-slate-50 min-h-screen">
        <form action="{{ route('admin.orders.update', $order->id) }}" method="POST" enctype="multipart/form-data">
            @csrf
            @method('PUT')
            <input type="hidden" name="redirect_to" value="{{ request('origin') }}">

            <!-- Header Section -->
            <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-8 gap-4">
                <div>
                    <h1 class="text-2xl font-bold text-slate-800">Order ID: {{ $order->order_number }}</h1>
                    <p class="text-slate-500 mt-1 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>
                        Placed Date: {{ $order->created_at->format('d/m/Y, h:i:s A') }}
                    </p>
                </div>
                <div class="flex items-center gap-3">
                    <a href="{{ route('admin.orders.index', request('origin') === 'incomplete' ? ['status' => 'incomplete'] : []) }}" class="px-4 py-2 bg-white border border-slate-300 text-slate-700 rounded-lg hover:bg-slate-50 font-medium transition-colors flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                        Cancel
                    </a>
                    <button type="submit" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 font-medium shadow-sm transition-colors flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>
                        Save
                    </button>
                </div>
            </div>

            @if(session('success'))
                <div class="mb-6 p-4 bg-green-50 border border-green-200 text-green-700 rounded-lg flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                    {{ session('success') }}
                </div>
            @endif

            @if($errors->any())
                <div class="mb-6 p-4 bg-red-50 border border-red-200 text-red-700 rounded-lg">
                    <ul class="list-disc list-inside">
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Left Column -->
                <div class="lg:col-span-2 space-y-6">
                    
                    <!-- Customer Information -->
                    <div class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                        <div class="px-6 py-4 border-b border-slate-100 flex items-center gap-2">
                            <svg class="w-5 h-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path></svg>
                            <h2 class="text-lg font-bold text-slate-800">Customer Information</h2>
                        </div>
                        <div class="p-6 grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-slate-700 mb-1">Customer Name *</label>
                                <input type="text" name="first_name" value="{{ old('first_name', $order->first_name) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-slate-700 mb-1">Mobile Number *</label>
                                <input type="text" name="phone" value="{{ old('phone', $order->phone) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm" required maxlength="11" pattern="\d{11}" title="Please enter exactly 11 digits" oninput="this.value = this.value.replace(/[^0-9]/g, '').slice(0, 11)">
                            </div>
                            <div class="md:col-span-2">
                                <label class="block text-sm font-medium text-slate-700 mb-1">Email Address</label>
                                <input type="email" name="email" value="{{ old('email', $order->email) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm">
                            </div>
                        </div>
                    </div>

                    <!-- Customer Address Information -->
                    <div class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                        <div class="px-6 py-4 border-b border-slate-100 flex items-center gap-2">
                            <svg class="w-5 h-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                            <h2 class="text-lg font-bold text-slate-800">Customer Address Information</h2>
                        </div>
                        <div class="p-6 space-y-6">
                            <div>
                                <label class="block text-sm font-medium text-slate-700 mb-3">Select Delivery Area *</label>
                                <div class="flex flex-wrap gap-6">
                                    <label class="flex items-center gap-2 cursor-pointer">
                                        <input type="radio" name="delivery_area" value="inside_dhaka" {{ old('delivery_area', $order->delivery_area) == 'inside_dhaka' ? 'checked' : '' }} class="w-4 h-4 text-blue-600 border-slate-300 focus:ring-blue-500">
                                        <span class="text-sm text-slate-700">Inside Dhaka (৳60 - ৳90)</span>
                                    </label>
                                    <label class="flex items-center gap-2 cursor-pointer">
                                        <input type="radio" name="delivery_area" value="outside_dhaka" {{ old('delivery_area', $order->delivery_area) == 'outside_dhaka' ? 'checked' : '' }} class="w-4 h-4 text-blue-600 border-slate-300 focus:ring-blue-500">
                                        <span class="text-sm text-slate-700">Outside Dhaka (৳130 - ৳170)</span>
                                    </label>
                                </div>
                            </div>
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-slate-700 mb-1">Select District (city)</label>
                                    <input type="text" name="district" id="district_input" list="district_list" value="{{ old('district', $order->district ?? $order->city) }}" placeholder="Type to search city" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm" autocomplete="off">
                                    <datalist id="district_list"></datalist>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-slate-700 mb-1">Select Thana (zone)</label>
                                    <input type="text" name="thana" id="thana_input" list="thana_list" value="{{ old('thana', $order->thana) }}" placeholder="Type to search Thana" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm" autocomplete="off">
                                    <datalist id="thana_list"></datalist>
                                </div>
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-slate-700 mb-1">Inset Address *</label>
                                <textarea name="address" rows="3" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm">{{ old('address', $order->address) }}</textarea>
                            </div>
                        </div>
                    </div>

                    <!-- Product Information -->
                    <div class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                        <div class="px-6 py-4 border-b border-slate-100 flex items-center justify-between">
                            <div class="flex items-center gap-2">
                                <svg class="w-5 h-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"></path></svg>
                                <h2 class="text-lg font-bold text-slate-800">Product Information & Pricing</h2>
                            </div>
                            <button type="button" class="bg-amber-500 hover:bg-amber-600 text-white rounded-full p-1 transition-colors">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg>
                            </button>
                        </div>
                        <div class="p-6 space-y-4">
                            <!-- Products List -->
                            <div id="orderItemsContainer" class="space-y-4">
                                @foreach($order->items as $item)
                                <div class="flex flex-col md:flex-row items-center gap-4 p-4 border border-slate-100 rounded-xl bg-slate-50/50" id="item-row-{{ $item->id }}">
                                    <div class="w-16 h-16 bg-white rounded-lg border border-slate-200 flex items-center justify-center overflow-hidden shrink-0">
                                        @if($item->product && $item->product->thumbnail)
                                            <img src="{{ Storage::url($item->product->thumbnail) }}" alt="" class="w-full h-full object-cover">
                                        @else
                                            <svg class="w-8 h-8 text-slate-300" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>
                                        @endif
                                    </div>
                                    
                                    <div class="flex-1 text-center md:text-left">
                                        <h3 class="font-bold text-slate-800 text-sm">{{ $item->product_name }}</h3>
                                        <p class="text-xs text-slate-500 mt-0.5">SKU: {{ $item->product->sku ?? 'N/A' }}</p>
                                        <p class="text-xs text-slate-500">
                                            @if($item->color) {{ $item->color->name }} @endif
                                            @if($item->size) - {{ $item->size->name }} @endif
                                        </p>
                                    </div>

                                    <div class="flex items-center gap-2">
                                        <div class="flex flex-col">
                                            <label class="text-[10px] text-slate-500">Update Price</label>
                                            <input type="number" name="items[{{ $item->id }}][price]" value="{{ $item->price }}" class="item-price w-20 rounded border-slate-300 py-1 text-sm focus:ring-blue-500" oninput="calculateTotal()">
                                        </div>
                                    </div>

                                    <div class="flex items-center gap-2">
                                        <div class="flex items-center border border-slate-300 rounded-lg bg-white">
                                            <button type="button" class="px-2 py-1 text-slate-500 hover:bg-slate-50 rounded-l-lg" onclick="this.nextElementSibling.stepDown(); this.nextElementSibling.dispatchEvent(new Event('input'))">-</button>
                                            <input type="number" name="items[{{ $item->id }}][quantity]" value="{{ $item->quantity }}" class="item-quantity w-12 text-center border-0 py-1 text-sm focus:ring-0" oninput="calculateTotal()">
                                            <button type="button" class="px-2 py-1 text-slate-500 hover:bg-slate-50 rounded-r-lg" onclick="this.previousElementSibling.stepUp(); this.previousElementSibling.dispatchEvent(new Event('input'))">+</button>
                                        </div>
                                    </div>

                                    <div class="item-total font-bold text-slate-800 w-24 text-right">
                                        = {{ number_format($item->price * $item->quantity, 0) }} Tk
                                    </div>

                                    <button type="button" onclick="document.getElementById('item-row-{{ $item->id }}').remove(); calculateTotal()" class="text-red-500 hover:text-red-600 bg-red-50 hover:bg-red-100 p-1.5 rounded-full transition-colors">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                                    </button>
                                </div>
                                @endforeach
                            </div>

                            <!-- Add Product Input -->
                            <div class="mt-4">
                                <label class="block text-sm font-medium text-slate-700 mb-1">Choose Another Product</label>
                                <div class="relative">
                                    <input type="text" readonly onclick="openProductModal()" placeholder="Click to select products..." class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm cursor-pointer bg-white">
                                    <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                        <svg class="w-5 h-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Right Column -->
                <div class="space-y-6">
                    
                    <!-- Payment methods & others -->
                    <div class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                        <div class="px-6 py-4 border-b border-slate-100 flex items-center gap-2">
                            <svg class="w-5 h-5 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"></path></svg>
                            <h2 class="text-lg font-bold text-slate-800">Payment methods & others</h2>
                        </div>
                        <div class="p-6 space-y-4">
                            <div class="p-3 bg-red-50 border border-red-100 rounded-lg">
                                <p class="text-xs font-medium text-red-800">Customer Payment Information</p>
                                <p class="text-sm text-red-600 mt-0.5">No payment information found</p>
                            </div>

                            <div class="grid grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-xs font-medium text-slate-700 mb-1">Payment Method</label>
                                    <select name="payment_method" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm py-2">
                                        <option value="cod" {{ old('payment_method', $order->payment_method) == 'cod' ? 'selected' : '' }}>Cash On Delivery</option>
                                        <option value="bkash" {{ old('payment_method', $order->payment_method) == 'bkash' ? 'selected' : '' }}>Bkash</option>
                                        <option value="nagad" {{ old('payment_method', $order->payment_method) == 'nagad' ? 'selected' : '' }}>Nagad</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-xs font-medium text-slate-700 mb-1">Advance Amount</label>
                                    <input type="number" name="advance_amount" id="advance_amount_input" value="{{ old('advance_amount', $order->advance_amount) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm py-2">
                                </div>
                            </div>

                            <div class="grid grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-xs font-medium text-slate-700 mb-1">Sender Number</label>
                                    <input type="text" name="payment_sender_number" value="{{ old('payment_sender_number', $order->payment_sender_number) }}" placeholder="Enter sender number" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm py-2">
                                </div>
                                <div>
                                    <label class="block text-xs font-medium text-slate-700 mb-1">Receiver Number</label>
                                    <input type="text" name="payment_receiver_number" value="{{ old('payment_receiver_number', $order->payment_receiver_number) }}" placeholder="Enter receiver number" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm py-2">
                                </div>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-slate-700 mb-1">Customer Order Note</label>
                                <textarea name="note" rows="2" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm">{{ old('note', $order->note) }}</textarea>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-slate-700 mb-1">Office Order Note</label>
                                <textarea name="staff_note" rows="2" placeholder="Enter office order note" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm">{{ old('staff_note', $order->staff_note) }}</textarea>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-slate-700 mb-1">Order Status</label>
                                <select name="status" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-sm py-2">
                                    <option value="pending" {{ $order->status == 'pending' ? 'selected' : '' }}>Pending</option>
                                    <option value="processing" {{ $order->status == 'processing' ? 'selected' : '' }}>Confirm</option>
                                    <option value="shipped" {{ $order->status == 'shipped' ? 'selected' : '' }}>Ready to Delivery</option>
                                    <option value="delivered" {{ $order->status == 'delivered' ? 'selected' : '' }}>Delivered</option>
                                    <option value="cancelled" {{ $order->status == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                                </select>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-slate-700 mb-1">Extra Document</label>
                                <input type="file" name="attachment" id="attachment_input" class="hidden" onchange="updateFileName(this)">
                                <div onclick="document.getElementById('attachment_input').click()" class="border-2 border-dashed border-slate-300 rounded-lg p-4 text-center hover:bg-slate-50 transition-colors cursor-pointer relative group">
                                    @if($order->attachment)
                                        <div class="absolute top-2 right-2 z-10">
                                            <a href="{{ Storage::url($order->attachment) }}" target="_blank" class="text-blue-600 hover:text-blue-800 bg-white rounded-full p-1 shadow-sm" title="View existing file" onclick="event.stopPropagation()">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path></svg>
                                            </a>
                                        </div>
                                    @endif
                                    
                                    <svg class="w-8 h-8 text-slate-400 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>
                                    <p class="text-xs text-slate-500" id="file_name_display">
                                        @if($order->attachment)
                                            File attached (Click to change)
                                        @else
                                            Click to Upload File
                                        @endif
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Total Pricing Calculation -->
                    <div class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                        <div class="px-6 py-4 border-b border-slate-100">
                            <h2 class="text-lg font-bold text-slate-800">Total Pricing Calculation</h2>
                        </div>
                        <div class="p-6 space-y-3">
                            <div class="flex justify-between items-center text-sm">
                                <span class="text-slate-600">Subtotal</span>
                                <span class="font-medium text-slate-800" id="subtotal_display">{{ number_format($order->items->sum(fn($i) => $i->price * $i->quantity), 2) }} Tk</span>
                            </div>
                            <div class="flex justify-between items-center text-sm">
                                <span class="text-slate-600">Delivery Cost</span>
                                <div class="flex items-center gap-1">
                                    <span class="text-slate-400 text-xs">(+)</span>
                                    <input type="number" name="delivery_cost" id="delivery_cost_input" value="{{ old('delivery_cost', $order->delivery_cost) }}" class="w-20 rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-xs py-1 text-right">
                                    <span class="font-medium text-slate-800">Tk</span>
                                </div>
                            </div>
                            <div class="flex justify-between items-center text-sm">
                                <span class="text-slate-600">Price (MRP)</span>
                                <span class="font-medium text-slate-800" id="mrp_display">{{ number_format($order->total_amount, 2) }} Tk</span>
                            </div>

                            <div class="border-t border-slate-100 my-3 pt-3">
                                <div class="flex justify-between items-center font-bold text-base mb-3">
                                    <span class="text-slate-800">Total Pricing Calculation</span>
                                    <span class="text-slate-800" id="total_calculation_display">{{ number_format($order->total_amount + $order->delivery_cost, 2) }} Tk</span>
                                </div>
                                <div class="flex justify-between items-center text-sm mb-3">
                                    <span class="text-slate-600">Advanced</span>
                                    <span class="font-medium text-slate-800" id="advanced_display">{{ number_format($order->advance_amount, 2) }} Tk</span>
                                </div>
                                
                                <div class="grid grid-cols-2 gap-4 mb-3">
                                    <div>
                                        <label class="block text-xs text-slate-500 mb-1">Extra Charge</label>
                                        <input type="number" name="extra_charge" id="extra_charge_input" value="{{ old('extra_charge', $order->extra_charge) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-xs py-1.5">
                                    </div>
                                    <div>
                                        <label class="block text-xs text-slate-500 mb-1">Additional Discount</label>
                                        <input type="number" name="discount_amount" id="discount_input" value="{{ old('discount_amount', $order->discount_amount) }}" class="w-full rounded-lg border-slate-300 focus:border-blue-500 focus:ring-blue-500 text-xs py-1.5">
                                    </div>
                                </div>

                                <div class="flex justify-between items-center font-bold text-lg pt-2 border-t border-slate-100">
                                    <span class="text-slate-800">Due Price</span>
                                    <span class="text-blue-600" id="due_price_display">{{ number_format($order->total_amount + $order->delivery_cost - $order->advance_amount + $order->extra_charge - $order->discount_amount, 2) }} Tk</span>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </form>
    </div>

    <!-- Product Selection Modal -->
    <div id="productSelectionModal" class="fixed inset-0 z-50 hidden overflow-y-auto" aria-labelledby="modal-title" role="dialog" aria-modal="true">
        <div class="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
            <!-- Backdrop -->
            <div id="modalBackdrop" class="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity duration-300 opacity-0" aria-hidden="true" onclick="closeProductModal()"></div>

            <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>

            <!-- Modal Panel -->
            <div id="modalPanel" class="inline-block align-middle bg-white rounded-2xl text-left overflow-hidden shadow-xl transform transition-all duration-300 scale-95 opacity-0 sm:my-8 sm:max-w-5xl sm:w-full">
                <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-lg leading-6 font-bold text-gray-900" id="modal-title">Select Products</h3>
                        <button type="button" onclick="closeProductModal()" class="text-gray-400 hover:text-gray-500 transition-transform hover:rotate-90 duration-200">
                            <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                        </button>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-12 gap-6 h-[550px]">
                        <!-- Available Products (1st Module) -->
                        <div class="md:col-span-7 border border-gray-200 rounded-xl flex flex-col h-full shadow-sm">
                            <div class="p-4 border-b border-gray-200 bg-gray-50 rounded-t-xl">
                                <h4 class="font-medium text-gray-700 mb-2">Available Products</h4>
                                <input type="text" id="productSearchInput" placeholder="Search product..." class="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm transition-shadow">
                            </div>
                            <div class="flex-1 overflow-y-auto p-4 space-y-3 custom-scrollbar" id="availableProductsList">
                                <!-- Products will be loaded here -->
                                <div class="text-center text-gray-500 mt-10">Start typing to search products...</div>
                            </div>
                        </div>

                        <!-- Selected Products (2nd Module - Smaller & Rounder) -->
                        <div class="md:col-span-5 border border-gray-200 rounded-2xl flex flex-col h-full shadow-md bg-blue-50/10">
                            <div class="p-4 border-b border-gray-200 bg-blue-50/50 rounded-t-2xl">
                                <h4 class="font-medium text-gray-800">Selected Products</h4>
                            </div>
                            <div class="flex-1 overflow-y-auto p-4 relative custom-scrollbar">
                                <div id="selectedProductsList" class="space-y-3"></div>
                                <div id="noSelectionMessage" class="text-center text-gray-500 mt-10 absolute inset-0 pointer-events-none flex items-center justify-center">
                                    <span class="bg-white/80 px-4 py-2 rounded-full shadow-sm">No products selected</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse border-t border-gray-100">
                    <button type="button" onclick="submitProductSelection()" class="w-full inline-flex justify-center rounded-xl border border-transparent shadow-sm px-4 py-2 bg-green-600 text-base font-medium text-white hover:bg-green-700 hover:shadow-md focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 sm:ml-3 sm:w-auto sm:text-sm transition-all transform hover:-translate-y-0.5">
                        Submit Changes
                    </button>
                    <button type="button" onclick="closeProductModal()" class="mt-3 w-full inline-flex justify-center rounded-xl border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 hover:text-gray-900 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm transition-colors">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

<style>
        .custom-scrollbar::-webkit-scrollbar {
            width: 6px;
        }
        .custom-scrollbar::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 8px;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 8px;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }
        @keyframes fade-in-up {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .animate-fade-in-up {
            animation: fade-in-up 0.3s ease-out forwards;
        }
        @keyframes slide-in-right {
            from { opacity: 0; transform: translateX(20px); }
            to { opacity: 1; transform: translateX(0); }
        }
        .animate-slide-in-right {
            animation: slide-in-right 0.3s ease-out forwards;
        }
    </style>
    <script>
        let selectedProducts = [];
        let searchTimeout;

        function openProductModal() {
            const modal = document.getElementById('productSelectionModal');
            const backdrop = document.getElementById('modalBackdrop');
            const panel = document.getElementById('modalPanel');
            
            modal.classList.remove('hidden');
            
            // Trigger animation
            requestAnimationFrame(() => {
                backdrop.classList.remove('opacity-0');
                panel.classList.remove('scale-95', 'opacity-0');
                panel.classList.add('scale-100', 'opacity-100');
            });

            document.getElementById('productSearchInput').focus();
            
            // Populate selectedProducts from the existing list if not already done
            if (selectedProducts.length === 0) {
                // Initialize with items from server-side variable
                const initialItems = @json($order->items);
                
                selectedProducts = initialItems.map(item => ({
                    id: item.product_id, // Important: use product_id
                    name: item.product_name,
                    price: parseFloat(item.price),
                    thumbnail: item.product ? (item.product.thumbnail ? '/storage/' + item.product.thumbnail : '') : '',
                    sku: item.product ? item.product.sku : '',
                    quantity: parseInt(item.quantity),
                    order_item_id: item.id // Keep track of existing item ID
                }));
                
                renderSelectedProducts();
            }

            // Load initial products
            fetchProducts();
        }

        function closeProductModal() {
            const modal = document.getElementById('productSelectionModal');
            const backdrop = document.getElementById('modalBackdrop');
            const panel = document.getElementById('modalPanel');

            backdrop.classList.add('opacity-0');
            panel.classList.remove('scale-100', 'opacity-100');
            panel.classList.add('scale-95', 'opacity-0');

            setTimeout(() => {
                modal.classList.add('hidden');
            }, 300); // Match transition duration
        }

        function fetchProducts(query = '') {
            const list = document.getElementById('availableProductsList');
            list.innerHTML = '<div class="text-center text-gray-500 mt-10"><i class="fas fa-spinner fa-spin"></i> Loading...</div>';

            fetch(`{{ route('admin.products.search') }}?q=${query}`)
                .then(response => response.json())
                .then(data => {
                    list.innerHTML = '';
                    
                    if (data.length === 0) {
                        list.innerHTML = '<div class="text-center text-gray-500 mt-10">No products found</div>';
                        return;
                    }

                    data.forEach(product => {
                        const isSelected = selectedProducts.some(p => p.id === product.id);
                        const btnText = isSelected ? 'Added' : 'Add';
                        const btnClass = isSelected ? 'bg-gray-400 cursor-not-allowed' : 'bg-blue-600 hover:bg-blue-700';
                        const disabled = isSelected ? 'disabled' : '';

                        const html = `
                            <div class="flex items-center gap-3 p-3 border border-gray-100 rounded-lg hover:bg-gray-50 transition-all duration-200 transform hover:-translate-y-0.5 shadow-sm hover:shadow-md animate-fade-in-up">
                                <div class="w-12 h-12 bg-gray-200 rounded overflow-hidden flex-shrink-0">
                                    ${product.thumbnail_url ? `<img src="${product.thumbnail_url}" class="w-full h-full object-cover">` : ''}
                                </div>
                                <div class="flex-1 min-w-0">
                                    <h5 class="text-sm font-medium text-gray-900 truncate">${product.name}</h5>
                                    <p class="text-xs text-gray-500">SKU: ${product.sku || 'N/A'}</p>
                                    <p class="text-sm font-bold text-gray-900 mt-0.5">${product.price} Tk</p>
                                </div>
                                <button type="button" onclick="addToSelection(${product.id}, '${product.name.replace(/'/g, "\\'")}', ${product.price}, '${product.thumbnail_url || ''}', '${product.sku || ''}')" 
                                    class="px-3 py-1.5 text-xs font-medium text-white rounded-lg shadow-sm transition-all duration-200 ${btnClass}" ${disabled} id="btn-add-${product.id}">
                                    ${btnText}
                                </button>
                            </div>
                        `;
                        list.insertAdjacentHTML('beforeend', html);
                    });
                });
        }

        document.getElementById('productSearchInput').addEventListener('input', function(e) {
            clearTimeout(searchTimeout);
            const query = e.target.value;
            
            searchTimeout = setTimeout(() => {
                fetchProducts(query);
            }, 300);
        });

        function addToSelection(id, name, price, thumbnail, sku) {
            if (selectedProducts.some(p => p.id === id)) return;

            selectedProducts.push({
                id, name, price, thumbnail, sku, quantity: 1, order_item_id: null
            });

            // Update Add Button
            const btn = document.getElementById(`btn-add-${id}`);
            if (btn) {
                btn.textContent = 'Added';
                btn.classList.remove('bg-blue-600', 'hover:bg-blue-700');
                btn.classList.add('bg-gray-400', 'cursor-not-allowed');
                btn.disabled = true;
            }

            renderSelectedProducts();
        }

        function removeFromSelection(index) {
            const product = selectedProducts[index];
            selectedProducts.splice(index, 1);
            
            // Update Add Button
            const btn = document.getElementById(`btn-add-${product.id}`);
            if (btn) {
                btn.textContent = 'Add';
                btn.classList.add('bg-blue-600', 'hover:bg-blue-700');
                btn.classList.remove('bg-gray-400', 'cursor-not-allowed');
                btn.disabled = false;
            }

            renderSelectedProducts();
        }

        function updateSelectionQuantity(index, change) {
            const product = selectedProducts[index];
            const newQty = product.quantity + change;
            if (newQty >= 1) {
                product.quantity = newQty;
                renderSelectedProducts();
            }
        }

        function renderSelectedProducts() {
            const list = document.getElementById('selectedProductsList');
            const msg = document.getElementById('noSelectionMessage');
            
            if (selectedProducts.length === 0) {
                list.innerHTML = '';
                msg.classList.remove('hidden');
                return;
            }

            msg.classList.add('hidden');
            list.innerHTML = '';

            selectedProducts.forEach((product, index) => {
                const html = `
                    <div class="flex items-center gap-3 p-3 border border-gray-100 rounded-lg bg-blue-50/30 transition-all duration-300 animate-slide-in-right hover:bg-blue-50/60 shadow-sm">
                        <div class="w-12 h-12 bg-gray-200 rounded overflow-hidden flex-shrink-0">
                            ${product.thumbnail ? `<img src="${product.thumbnail}" class="w-full h-full object-cover">` : ''}
                        </div>
                        <div class="flex-1 min-w-0">
                            <h5 class="text-sm font-medium text-gray-900 truncate">${product.name}</h5>
                            <p class="text-xs text-gray-500">${product.price} Tk</p>
                        </div>
                        <div class="flex items-center border border-gray-300 rounded bg-white">
                            <button type="button" onclick="updateSelectionQuantity(${index}, -1)" class="px-2 py-1 text-gray-600 hover:bg-gray-100 transition-colors">-</button>
                            <span class="px-2 py-1 text-sm font-medium text-gray-900 min-w-[30px] text-center">${product.quantity}</span>
                            <button type="button" onclick="updateSelectionQuantity(${index}, 1)" class="px-2 py-1 text-gray-600 hover:bg-gray-100 transition-colors">+</button>
                        </div>
                        <button type="button" onclick="removeFromSelection(${index})" class="p-1.5 text-red-500 hover:bg-red-50 rounded-full transition-colors transform hover:scale-110">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                        </button>
                    </div>
                `;
                list.insertAdjacentHTML('beforeend', html);
            });
        }

        function submitProductSelection() {
            const container = document.getElementById('orderItemsContainer');
            container.innerHTML = ''; // Clear existing items

            selectedProducts.forEach((product, index) => {
                const rowId = product.order_item_id ? `item-row-${product.order_item_id}` : `new-item-row-${index}`;
                
                // Determine input names
                // If it has order_item_id, it's an existing item: items[id][field]
                // If not, it's a new item: new_items[temp_id][field]
                const inputNamePrefix = product.order_item_id 
                    ? `items[${product.order_item_id}]` 
                    : `new_items[${Date.now() + index}]`;

                const html = `
                    <div class="flex flex-col md:flex-row items-center gap-4 p-4 border border-slate-100 rounded-xl bg-slate-50/50" id="${rowId}">
                        <div class="w-16 h-16 bg-white rounded-lg border border-slate-200 flex items-center justify-center overflow-hidden shrink-0">
                            ${product.thumbnail ? `<img src="${product.thumbnail}" class="w-full h-full object-cover">` : 
                                `<svg class="w-8 h-8 text-slate-300" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>`}
                        </div>
                        
                        <div class="flex-1 text-center md:text-left">
                            <h3 class="font-bold text-slate-800 text-sm">${product.name}</h3>
                            <p class="text-xs text-slate-500 mt-0.5">SKU: ${product.sku || 'N/A'}</p>
                            <!-- Hidden inputs for product info if needed, though backend mostly needs ID and qty/price -->
                            ${!product.order_item_id ? `<input type="hidden" name="${inputNamePrefix}[product_id]" value="${product.id}">` : ''}
                        </div>

                        <div class="flex items-center gap-2">
                            <div class="flex flex-col">
                                <label class="text-[10px] text-slate-500">Update Price</label>
                                <input type="number" name="${inputNamePrefix}[price]" value="${product.price}" class="item-price w-20 rounded border-slate-300 py-1 text-sm focus:ring-blue-500" oninput="calculateTotal()">
                            </div>
                        </div>

                        <div class="flex items-center gap-2">
                            <div class="flex items-center border border-slate-300 rounded-lg bg-white">
                                <button type="button" class="px-2 py-1 text-slate-500 hover:bg-slate-50 rounded-l-lg" onclick="this.nextElementSibling.stepDown(); this.nextElementSibling.dispatchEvent(new Event('input'))">-</button>
                                <input type="number" name="${inputNamePrefix}[quantity]" value="${product.quantity}" class="item-quantity w-12 text-center border-0 py-1 text-sm focus:ring-0" oninput="calculateTotal()">
                                <button type="button" class="px-2 py-1 text-slate-500 hover:bg-slate-50 rounded-r-lg" onclick="this.previousElementSibling.stepUp(); this.previousElementSibling.dispatchEvent(new Event('input'))">+</button>
                            </div>
                        </div>

                        <div class="item-total font-bold text-slate-800 w-24 text-right">
                            = ${(product.price * product.quantity).toFixed(0)} Tk
                        </div>

                        <button type="button" onclick="this.closest('.flex').remove(); calculateTotal(); removeFromSelectedProducts(${product.id})" class="text-red-500 hover:text-red-600 bg-red-50 hover:bg-red-100 p-1.5 rounded-full transition-colors">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                        </button>
                    </div>
                `;
                container.insertAdjacentHTML('beforeend', html);
            });

            closeProductModal();
            calculateTotal(); // Trigger total recalculation
        }

        // Helper to remove from selectedProducts if removed from main list (optional, keeps state consistent if modal reopened)
        function removeFromSelectedProducts(productId) {
            const index = selectedProducts.findIndex(p => p.id === productId);
            if (index > -1) {
                selectedProducts.splice(index, 1);
            }
        }

        function updateFileName(input) {
            const display = document.getElementById('file_name_display');
            if (input.files && input.files[0]) {
                display.textContent = input.files[0].name;
                display.classList.add('text-green-600', 'font-medium');
                display.classList.remove('text-slate-500');
            }
        }
    </script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const itemPrices = document.querySelectorAll('.item-price');
            const itemQuantities = document.querySelectorAll('.item-quantity');
            const itemTotals = document.querySelectorAll('.item-total');
            
            const subtotalDisplay = document.getElementById('subtotal_display');
            const deliveryCostInput = document.getElementById('delivery_cost_input');
            const mrpDisplay = document.getElementById('mrp_display');
            const totalCalculationDisplay = document.getElementById('total_calculation_display');
            const advanceAmountInput = document.getElementById('advance_amount_input');
            const advancedDisplay = document.getElementById('advanced_display');
            const extraChargeInput = document.getElementById('extra_charge_input');
            const discountInput = document.getElementById('discount_input');
            const duePriceDisplay = document.getElementById('due_price_display');

            // Delivery Area Cost Update
            const deliveryAreaRadios = document.querySelectorAll('input[name="delivery_area"]');
            
            deliveryAreaRadios.forEach(radio => {
                radio.addEventListener('change', function() {
                    if (this.checked) {
                        if (this.value === 'inside_dhaka') {
                            if(deliveryCostInput) {
                                deliveryCostInput.value = 60;
                                deliveryCostInput.dispatchEvent(new Event('input')); // Trigger calculation
                            }
                        } else if (this.value === 'outside_dhaka') {
                            if(deliveryCostInput) {
                                deliveryCostInput.value = 130;
                                deliveryCostInput.dispatchEvent(new Event('input')); // Trigger calculation
                            }
                        }
                    }
                });
            });

            function formatCurrency(amount) {
                return new Intl.NumberFormat('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(amount) + ' Tk';
            }

            function parseValue(value) {
                return parseFloat(value) || 0;
            }

            function calculateTotals() {
                // Re-query dynamic elements
                const itemPrices = document.querySelectorAll('.item-price');
                const itemQuantities = document.querySelectorAll('.item-quantity');
                const itemTotals = document.querySelectorAll('.item-total');

                let subtotal = 0;

                // Calculate product row totals
                itemPrices.forEach((priceInput, index) => {
                    const price = parseValue(priceInput.value);
                    const quantity = itemQuantities[index] ? parseValue(itemQuantities[index].value) : 0;
                    const total = price * quantity;
                    
                    subtotal += total;
                    if(itemTotals[index]) {
                        itemTotals[index].textContent = '= ' + formatCurrency(total);
                    }
                });

                // Update Subtotal Display
                if(subtotalDisplay) subtotalDisplay.textContent = formatCurrency(subtotal);
                if(mrpDisplay) mrpDisplay.textContent = formatCurrency(subtotal);

                // Get other values
                const deliveryCost = parseValue(deliveryCostInput ? deliveryCostInput.value : 0);
                const advanceAmount = parseValue(advanceAmountInput ? advanceAmountInput.value : 0);
                const extraCharge = parseValue(extraChargeInput ? extraChargeInput.value : 0);
                const discount = parseValue(discountInput ? discountInput.value : 0);

                // Calculate Totals
                const totalWithDelivery = subtotal + deliveryCost;
                const duePrice = totalWithDelivery + extraCharge - advanceAmount - discount;

                // Update Displays
                if(totalCalculationDisplay) totalCalculationDisplay.textContent = formatCurrency(totalWithDelivery);
                if(advancedDisplay) advancedDisplay.textContent = formatCurrency(advanceAmount);
                if(duePriceDisplay) duePriceDisplay.textContent = formatCurrency(duePrice);
            }

            // Expose globally for inline handlers
            window.calculateTotal = calculateTotals;

            // Attach Event Listeners
            const allInputs = [
                ...itemPrices, 
                ...itemQuantities, 
                deliveryCostInput, 
                advanceAmountInput, 
                extraChargeInput, 
                discountInput
            ];

            allInputs.forEach(input => {
                if(input) {
                    input.addEventListener('input', calculateTotals);
                }
            });

            // Initial Calculation (optional, to ensure consistency on load)
            // calculateTotals(); 

            // Bangladesh Location Data
            const bdLocations = {
                "Bagerhat": ["Bagerhat Sadar", "Chitalmari", "Fakirhat", "Kachua", "Mollahat", "Mongla", "Morrelganj", "Rampal", "Sarankhola"],
                "Bandarban": ["Ali Kadam", "Bandarban Sadar", "Lama", "Naikhongchhari", "Rowangchhari", "Ruma", "Thanchi"],
                "Barguna": ["Amtali", "Bamna", "Barguna Sadar", "Betagi", "Patharghata", "Taltali"],
                "Barishal": ["Agailjhara", "Babuganj", "Bakerganj", "Banaripara", "Barishal Sadar", "Gaurnadi", "Hizla", "Mehendiganj", "Muladi", "Wazirpur"],
                "Bhola": ["Bhola Sadar", "Burhanuddin", "Char Fasson", "Daulatkhan", "Lalmohan", "Manpura", "Tazumuddin"],
                "Bogura": ["Adamdighi", "Bogura Sadar", "Dhunat", "Dhupchanchia", "Gabtali", "Kahaloo", "Nandigram", "Sariakandi", "Sherpur", "Shibganj", "Sonatala"],
                "Brahmanbaria": ["Akhaura", "Ashuganj", "Bancharampur", "Bijoynagar", "Brahmanbaria Sadar", "Kasba", "Nabinagar", "Nasirnagar", "Sarail"],
                "Chandpur": ["Chandpur Sadar", "Faridganj", "Haimchar", "Haziganj", "Kachua", "Matlab Dakshin", "Matlab Uttar", "Shahrasti"],
                "Chattogram": ["Anwara", "Banshkhali", "Boalkhali", "Chandanaish", "Chattogram City", "Fatikchhari", "Hathazari", "Karnaphuli", "Lohagara", "Mirsharai", "Patiya", "Rangunia", "Raozan", "Sandwip", "Satkania", "Sitakunda"],
                "Chuadanga": ["Alamdanga", "Chuadanga Sadar", "Damurhuda", "Jibannagar"],
                "Cox's Bazar": ["Chakaria", "Cox's Bazar Sadar", "Kutubdia", "Maheshkhali", "Pekua", "Ramu", "Teknaf", "Ukhia"],
                "Cumilla": ["Barura", "Brahmanpara", "Burichang", "Chandina", "Chauddagram", "Cumilla Sadar", "Cumilla Sadar Dakshin", "Daudkandi", "Debidwar", "Homna", "Laksam", "Lalmai", "Manoharganj", "Meghna", "Muradnagar", "Nangalkot", "Titas"],
                "Dhaka": ["Adabor", "Badda", "Bangshal", "Bimanbandar", "Cantonment", "Chawkbazar", "Dakshinkhan", "Darus Salam", "Demra", "Dhamrai", "Dhanmondi", "Dohar", "Gendaria", "Gulshan", "Hazaribagh", "Jatrabari", "Kadamtali", "Kafrul", "Kalabagan", "Kamrangirchar", "Keraniganj", "Khilgaon", "Khilkhet", "Kotwali", "Lalbagh", "Mirpur", "Mohammadpur", "Motijheel", "Nawabganj", "New Market", "Pallabi", "Paltan", "Ramna", "Rampura", "Sabujbagh", "Savar", "Shah Ali", "Shahbag", "Sher-e-Bangla Nagar", "Shyampur", "Sutrapur", "Tejgaon", "Tejgaon Industrial Area", "Turag", "Uttara", "Uttar Khan", "Vatara", "Wari"],
                "Dinajpur": ["Birampur", "Birganj", "Biral", "Bochaganj", "Chirirbandar", "Dinajpur Sadar", "Ghoraghat", "Hakimpur", "Kaharole", "Khansama", "Nawabganj", "Parbatipur", "Phulbari"],
                "Faridpur": ["Alfadanga", "Bhanga", "Boalmari", "Charbhadrasan", "Faridpur Sadar", "Madhukhali", "Nagarkanda", "Sadarpur", "Saltha"],
                "Feni": ["Chhagalnaiya", "Daganbhuiyan", "Feni Sadar", "Fulgazi", "Parshuram", "Sonagazi"],
                "Gaibandha": ["Fulchhari", "Gaibandha Sadar", "Gobindaganj", "Palashbari", "Sadullapur", "Saghata", "Sundarganj"],
                "Gazipur": ["Gazipur Sadar", "Kaliakair", "Kaliganj", "Kapasia", "Sreepur"],
                "Gopalganj": ["Gopalganj Sadar", "Kashiani", "Kotalipara", "Muksudpur", "Tungipara"],
                "Habiganj": ["Ajmiriganj", "Bahubal", "Baniyachong", "Chunarughat", "Habiganj Sadar", "Lakhai", "Madhabpur", "Nabiganj", "Shaistaganj"],
                "Jamalpur": ["Bakshiganj", "Dewanganj", "Islampur", "Jamalpur Sadar", "Madarganj", "Melandaha", "Sarishabari"],
                "Jashore": ["Abhaynagar", "Bagherpara", "Chaugachha", "Jashore Sadar", "Jhikargachha", "Keshabpur", "Manirampur", "Sharsha"],
                "Jhalokati": ["Jhalokati Sadar", "Kathalia", "Nalchity", "Rajapur"],
                "Jhenaidah": ["Harinakunda", "Jhenaidah Sadar", "Kaliganj", "Kotchandpur", "Maheshpur", "Shailkupa"],
                "Joypurhat": ["Akkelpur", "Joypurhat Sadar", "Kalai", "Khetlal", "Panchbibi"],
                "Khagrachhari": ["Dighinala", "Guimara", "Khagrachhari Sadar", "Lakshmichhari", "Mahalchhari", "Manikchhari", "Matiranga", "Panchhari", "Ramgarh"],
                "Khulna": ["Batiaghata", "Dacope", "Dumuria", "Dighalia", "Khulna Sadar", "Koyra", "Paikgachha", "Phultala", "Rupsha", "Terokhada"],
                "Kishoreganj": ["Austagram", "Bajitpur", "Bhairab", "Hossainpur", "Itna", "Karimganj", "Katiadi", "Kishoreganj Sadar", "Kuliarchar", "Mithamain", "Nikli", "Pakundia", "Tarail"],
                "Kurigram": ["Bhurungamari", "Char Rajibpur", "Chilmari", "Kurigram Sadar", "Nageshwari", "Phulbari", "Rajarhat", "Raomari", "Ulipur"],
                "Kushtia": ["Bheramara", "Daulatpur", "Khoksa", "Kumarkhali", "Kushtia Sadar", "Mirpur"],
                "Lakshmipur": ["Kamalnagar", "Lakshmipur Sadar", "Raipur", "Ramganj", "Ramgati"],
                "Lalmonirhat": ["Aditmari", "Hatibandha", "Kaliganj", "Lalmonirhat Sadar", "Patgram"],
                "Madaripur": ["Kalkini", "Madaripur Sadar", "Rajoir", "Shibchar"],
                "Magura": ["Magura Sadar", "Mohammadpur", "Shalikha", "Sreepur"],
                "Manikganj": ["Daulatpur", "Ghior", "Harirampur", "Manikganj Sadar", "Saturia", "Shivalaya", "Singair"],
                "Meherpur": ["Gangni", "Meherpur Sadar", "Mujibnagar"],
                "Moulvibazar": ["Barlekha", "Juri", "Kamalganj", "Kulaura", "Moulvibazar Sadar", "Rajnagar", "Sreemangal"],
                "Munshiganj": ["Gazaria", "Lohajang", "Munshiganj Sadar", "Sirajdikhan", "Sreenagar", "Tongibari"],
                "Mymensingh": ["Bhaluka", "Dhobaura", "Fulbaria", "Gafargaon", "Gauripur", "Haluaghat", "Ishwarganj", "Mymensingh Sadar", "Muktagachha", "Nandail", "Phulpur", "Tara Khanda", "Trishal"],
                "Naogaon": ["Atrai", "Badalgachhi", "Dhamoirhat", "Manda", "Mohadevpur", "Naogaon Sadar", "Niamatpur", "Patnitala", "Porsha", "Raninagar", "Sapahar"],
                "Narail": ["Kalia", "Lohagara", "Narail Sadar"],
                "Narayanganj": ["Araihazar", "Bandar", "Narayanganj Sadar", "Rupganj", "Sonargaon"],
                "Narsingdi": ["Belabo", "Monohardi", "Narsingdi Sadar", "Palash", "Raipura", "Shibpur"],
                "Natore": ["Bagatipara", "Baraigram", "Gurudaspur", "Lalpur", "Natore Sadar", "Naldanga", "Singra"],
                "Netrokona": ["Atpara", "Barhatta", "Durgapur", "Khaliajuri", "Kalmakanda", "Kendua", "Madan", "Mohanganj", "Netrokona Sadar", "Purbadhala"],
                "Nilphamari": ["Dimla", "Domar", "Jaldhaka", "Kishoreganj", "Nilphamari Sadar", "Saidpur"],
                "Noakhali": ["Begumganj", "Chatkhil", "Companiganj", "Hatiya", "Kabirhat", "Noakhali Sadar", "Senbagh", "Sonaimuri", "Subarnachar"],
                "Pabna": ["Atgharia", "Bera", "Bhangura", "Chatmohar", "Faridpur", "Ishwardi", "Pabna Sadar", "Santhia", "Sujanagar"],
                "Panchagarh": ["Atwari", "Boda", "Debiganj", "Panchagarh Sadar", "Tetulia"],
                "Patuakhali": ["Bauphal", "Dashmina", "Dumki", "Galachipa", "Kalapara", "Mirzaganj", "Patuakhali Sadar", "Rangabali"],
                "Pirojpur": ["Bhandaria", "Kawkhali", "Mathbaria", "Nazirpur", "Nesarabad", "Pirojpur Sadar", "Zianagar"],
                "Rajbari": ["Baliakandi", "Goalandaghat", "Kalukhali", "Pangsha", "Rajbari Sadar"],
                "Rajshahi": ["Bagha", "Bagmara", "Charghat", "Durgapur", "Godagari", "Mohanpur", "Paba", "Puthia", "Rajshahi Sadar", "Tanore"],
                "Rangamati": ["Bagaichhari", "Barkal", "Belaichhari", "Juraichhari", "Kaptai", "Kaukhali", "Langadu", "Naniarchar", "Rajasthali", "Rangamati Sadar"],
                "Rangpur": ["Badarganj", "Gangachhara", "Kaunia", "Mithapukur", "Pirgachha", "Pirganj", "Rangpur Sadar", "Taraganj"],
                "Satkhira": ["Assasuni", "Debhata", "Kalaroa", "Kaliganj", "Satkhira Sadar", "Shyamnagar", "Tala"],
                "Shariatpur": ["Bhedarganj", "Damudya", "Gosairhat", "Naria", "Shariatpur Sadar", "Zajira"],
                "Sherpur": ["Jhenaigati", "Nakla", "Nalitabari", "Sherpur Sadar", "Sreebardi"],
                "Sirajganj": ["Belkuchi", "Chauhali", "Kamarkhanda", "Kazipur", "Raiganj", "Shahjadpur", "Sirajganj Sadar", "Tarash", "Ullahpara"],
                "Sunamganj": ["Bishwamvarpur", "Chhatak", "Dakshin Sunamganj", "Derai", "Dharamapasha", "Dowarabazar", "Jagannathpur", "Jamalganj", "Sullah", "Sunamganj Sadar", "Tahirpur"],
                "Sylhet": ["Balaganj", "Beanibazar", "Bishwanath", "Companiganj", "Dakshin Surma", "Fenchuganj", "Golapganj", "Gowainghat", "Jaintiapur", "Kanaighat", "Osmani Nagar", "Sylhet Sadar", "Zakiganj"],
                "Tangail": ["Basail", "Bhuapur", "Delduar", "Dhanbari", "Gopalpur", "Ghatail", "Kalihati", "Madhupur", "Mirzapur", "Nagarpur", "Sakhipur", "Tangail Sadar"],
                "Thakurgaon": ["Baliadangi", "Haripur", "Pirganj", "Ranisankail", "Thakurgaon Sadar"]
            };

            const districtInput = document.getElementById('district_input');
            const thanaInput = document.getElementById('thana_input');
            const districtList = document.getElementById('district_list');
            const thanaList = document.getElementById('thana_list');

            // Populate District List
            if(districtList) {
                Object.keys(bdLocations).sort().forEach(district => {
                    const option = document.createElement('option');
                    option.value = district;
                    districtList.appendChild(option);
                });
            }

            // Function to update Thana list based on selected district
            function updateThanaList(selectedDistrict) {
                if(thanaList) {
                    thanaList.innerHTML = ''; // Clear existing options
                    if(bdLocations[selectedDistrict]) {
                        bdLocations[selectedDistrict].sort().forEach(thana => {
                            const option = document.createElement('option');
                            option.value = thana;
                            thanaList.appendChild(option);
                        });
                    }
                }
            }

            // Event Listener for District Change
            if(districtInput) {
                districtInput.addEventListener('input', function() {
                    updateThanaList(this.value);
                    // Clear thana input if district changes
                    // thanaInput.value = ''; 
                });

                // Initialize Thana list if district is already selected (e.g. on edit)
                if(districtInput.value) {
                    updateThanaList(districtInput.value);
                }
            } 
        });
    </script>
</x-admin-layout>
