<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Color;
use App\Models\Size;
use App\Services\FraudCheckService;
use App\Services\SmsService;
use App\Services\FacebookConversionService;
use App\Services\TiktokConversionService;
use App\Models\ShippingCharge;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class CheckoutController extends Controller
{
    public function index(Request $request)
    {
        $cart = [];

        // Handle Direct Checkout (Buy Now) from URL parameters
        if ($request->has('product_id')) {
            $product = Product::find($request->product_id);
            if ($product) {
                $price = $product->offer_price ?? $product->price;
                $variant = null;

                // Check for variation
                $colorName = null;
                $sizeName = null;

                if ($request->filled('color_id')) {
                    $color = Color::find($request->color_id);
                    if ($color) $colorName = $color->name;
                }
                if ($request->filled('size_id')) {
                    $size = Size::find($request->size_id);
                    if ($size) $sizeName = $size->name;
                }

                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $price,
                    'quantity' => $request->quantity ?? 1,
                    'thumbnail' => $variant && $variant->image ? $variant->image : $product->thumbnail,
                    'color_id' => $request->filled('color_id') ? $request->color_id : null,
                    'size_id' => $request->filled('size_id') ? $request->size_id : null,
                    'color_name' => $colorName,
                    'size_name' => $sizeName,
                ];
            }
        } else {
            // Normal Cart Checkout
            $cart = Session::get('cart', []);
        }

        if (count($cart) == 0) {
            return redirect()->route('shop')->with('error', 'Your cart is empty');
        }

        $total = 0;
        $tiktokContents = [];
        foreach ($cart as $key => $item) {
            $total += $item['price'] * $item['quantity'];
            
            $productId = $item['product_id'] ?? $key;
            if (!isset($item['product_id']) && strpos($key, '_') !== false) {
               $productId = explode('_', $key)[0];
            }
            $tiktokContents[] = [
               'content_id' => (string)$productId,
               'quantity' => $item['quantity'],
               'price' => $item['price'],
               'content_name' => $item['name'] ?? 'Product',
            ];
        }

        $settings = \App\Models\GeneralSetting::first();
        $shippingCharges = ShippingCharge::where('status', true)->get();
        
        $eventId = (string) Str::uuid();

        // Fire TikTok & Facebook CAPI InitiateCheckout Event
        try {
            $user = Auth::user();
            $userData = [
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ];
            
            if ($user) {
                $userData['email'] = $user->email;
                $userData['phone'] = $user->phone;
                $userData['external_id'] = (string)$user->id;
            }

            $tiktokProperties = [
                'currency' => 'BDT',
                'value' => $total,
                'content_type' => 'product',
                'contents' => $tiktokContents,
            ];

            // TikTok
            app(TiktokConversionService::class)->track(
                'InitiateCheckout',
                $userData,
                $tiktokProperties,
                $eventId
            );

            // Facebook
            $facebookContents = [];
            foreach ($tiktokContents as $item) {
                $facebookContents[] = [
                    'id' => $item['content_id'],
                    'quantity' => $item['quantity'],
                    'item_price' => $item['price'],
                    'title' => $item['content_name']
                ];
            }

            $facebookCustomData = [
                'currency' => 'BDT',
                'value' => $total,
                'content_type' => 'product',
                'contents' => $facebookContents,
                'num_items' => count($facebookContents)
            ];

            app(FacebookConversionService::class)->track(
                'InitiateCheckout',
                $userData,
                $facebookCustomData,
                $eventId
            );
        } catch (\Exception $e) {
            \Log::error('CAPI InitiateCheckout Event Failed: ' . $e->getMessage());
        }

        return view('frontend.checkout.index', compact('cart', 'total', 'settings', 'eventId', 'shippingCharges'));
    }

    public function calculateShipping(Request $request)
    {
        $request->validate([
            'area_id' => 'required|exists:shipping_charges,id',
        ]);

        $areaId = $request->input('area_id');
        
        // Prepare Cart
        $cart = [];
        if ($request->has('product_id')) {
             $product = Product::find($request->product_id);
             if ($product) {
                $price = $product->offer_price ?? $product->price;
                $variant = null;

                // Check for variation logic for shipping calculation
                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'quantity' => $request->quantity ?? 1,
                    // We need correct price for total calculation in response
                    'price' => $price,
                    'color_id' => $request->color_id,
                    'size_id' => $request->size_id,
                ];
             }
        } else {
             $cart = Session::get('cart', []);
        }

        $cost = $this->_calculateShippingCost($areaId, $cart);
        
        $subtotal = 0;
        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? explode('_', $key)[0];
            $product = Product::find($productId);
            
            $quantity = $item['quantity'];
            $price = isset($item['price']) ? $item['price'] : ($product->price ?? 0);
            if (isset($item['price'])) $subtotal += $price * $quantity;
            else $subtotal += ($product ? $product->offer_price ?? $product->price : 0) * $quantity;
        }
        
        return response()->json([
            'cost' => $cost, 
            'total' => $subtotal + $cost,
            'formatted_cost' => '৳' . number_format($cost, 0),
            'formatted_total' => '৳' . number_format($subtotal + $cost, 0)
        ]);
    }

    private function _calculateShippingCost($areaId, $cart)
    {
        $shippingCharge = ShippingCharge::find($areaId);
        if (!$shippingCharge) {
            return 0;
        }

        $totalWeight = 0;
        $extraShippingCost = 0;

        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? explode('_', $key)[0];
            $product = Product::find($productId);
            
            $quantity = $item['quantity'];

                if ($product) {
                $weight = $product->weight;
                $weightUnit = $product->weight_unit;
                $variantShippingCost = null;
                $deliveryType = 'flat_rate'; // Default

                // Check for variant
                if (isset($item['color_id']) || isset($item['size_id'])) {
                    $variantQuery = ProductVariant::where('product_id', $productId);
                    if (isset($item['color_id'])) {
                        $variantQuery->where('color_id', $item['color_id']);
                    }
                    if (isset($item['size_id'])) {
                        $variantQuery->where('size_id', $item['size_id']);
                    }
                    $variant = $variantQuery->first();

                    if ($variant) {
                        if ($variant->weight) {
                            $weight = $variant->weight;
                            $weightUnit = $variant->weight_unit;
                        }
                        if ($variant->shipping_cost) {
                            $variantShippingCost = $variant->shipping_cost;
                        }
                        if ($variant->delivery_type) {
                            $deliveryType = $variant->delivery_type;
                        }
                    }
                }

                if ($deliveryType == 'free' || $deliveryType == 'pickup') {
                    // No shipping cost for this item
                    continue; 
                }

                if ($variantShippingCost !== null && $variantShippingCost > 0) {
                     $extraShippingCost += ($variantShippingCost * $quantity);
                } else {
                    if ($weight) {
                        // Normalize to KG
                        if (in_array(strtolower($weightUnit), ['gm', 'g', 'gram', 'grams'])) {
                             $weight = $weight / 1000;
                        }
                        $totalWeight += ($weight * $quantity);
                    }
                }
            }
        }
        
        // Default weight if 0 (e.g. 0.5kg base) ONLY if there are items contributing to weight
        // If all items have custom shipping cost, totalWeight might be 0.
        
        $weightCost = 0;
        if ($totalWeight > 0) {
             if ($totalWeight <= 0.5) {
                $weightCost = $shippingCharge->charge_0_0_5;
            } elseif ($totalWeight <= 1) {
                $weightCost = $shippingCharge->charge_0_5_1;
            } elseif ($totalWeight <= 2) {
                $weightCost = $shippingCharge->charge_1_2;
            } else {
                // Over 2kg logic: Base charge (Charge for 2kg range) + Extra per kg
                $weightCost = $shippingCharge->charge_1_2 + (ceil($totalWeight - 2) * $shippingCharge->charge_over_2);
            }
        } elseif ($extraShippingCost == 0 && count($cart) > 0) {
            // Fallback if weight is 0 and no extra cost (maybe product has no weight set)
             $weightCost = $shippingCharge->charge_0_0_5;
        }
        
        return $weightCost + $extraShippingCost;
    }

    public function storeIncomplete(Request $request)
    {
        // Custom validation for Bangladeshi phone number
        // Must be 11 digits and start with 013, 014, 015, 016, 017, 018, 019
        if (!preg_match('/^01[3-9]\d{8}$/', $request->phone)) {
            return response()->json(['status' => 'ignored']);
        }

        // Check if the user has already placed a real order in the last 15 minutes.
        // This prevents creating an "incomplete" order immediately after a successful checkout
        // due to race conditions or lingering frontend events.
        $recentRealOrder = Order::where('phone', $request->phone)
            ->where('status', '!=', 'incomplete')
            ->where('status', '!=', 'cancelled')
            ->where('created_at', '>=', now()->subMinutes(15))
            ->first();

        if ($recentRealOrder) {
            // If a real order exists, ensure no incomplete orders hang around
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            return response()->json(['status' => 'ignored_recent_purchase']);
        }

        $request->validate([
            'phone' => 'required|string|max:20',
        ]);

        $cart = [];
        if ($request->has('product_id')) {
             $product = Product::find($request->product_id);
             if ($product) {
                $price = $product->offer_price ?? $product->price;
                
                // Variation Check for Incomplete Order
                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'quantity' => $request->quantity ?? 1,
                    'price' => $price,
                    'name' => $product->name,
                    'color_id' => $request->filled('color_id') ? $request->color_id : null,
                    'size_id' => $request->filled('size_id') ? $request->size_id : null,
                ];
             }
        } else {
             $cart = Session::get('cart', []);
        }

        if (count($cart) == 0) {
            return response()->json(['status' => 'empty_cart']);
        }

        // Calculate Shipping
        $shippingCost = 0;
        if ($request->has('area_id')) {
            $shippingCost = $this->_calculateShippingCost($request->area_id, $cart);
        }

        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        $total += $shippingCost;

        // 1. Try to find existing incomplete order by Input ID (Most reliable for edits)
        // Then by Session
        $orderId = $request->input('incomplete_order_id') ?? Session::get('incomplete_order_id');
        $order = null;

        if ($orderId) {
            $order = Order::where('id', $orderId)->where('status', 'incomplete')->first();
        }

        // 2. If not found by session, try by Phone (Backup)
        if (!$order) {
            $existingOrders = Order::where('phone', $request->phone)
                          ->where('status', 'incomplete')
                          ->orderBy('id', 'desc')
                          ->get();

            if ($existingOrders->count() > 0) {
                $order = $existingOrders->first();
                // Delete older incomplete orders for this phone to prevent duplicates
                if ($existingOrders->count() > 1) {
                    Order::whereIn('id', $existingOrders->slice(1)->pluck('id'))->delete();
                }
            } else {
                $order = new Order();
                $order->order_number = 'INC-' . strtoupper(Str::random(10));
                $order->status = 'incomplete';
            }
        }

        $order->user_id = Auth::id();
        $order->first_name = $request->first_name ?? '';
        $order->last_name = $request->last_name ?? '';
        $order->email = $request->email ?? '';
        $order->phone = $request->phone;
        $order->address = $request->address ?? '';
        $order->city = $request->city ?? '';
        $order->postal_code = $request->postal_code ?? '';
        $order->delivery_cost = $shippingCost;
        $order->total_amount = $total;
        $order->payment_method = $request->payment_method ?? 'cod';
        $order->payment_status = 'pending';
        $order->source = $request->source ?? 'website';
        $order->save();

        // Store ID in session to track this specific incomplete order
        Session::put('incomplete_order_id', $order->id);

        // Ensure no other incomplete orders exist for this phone (cleanup duplicates)
        Order::where('phone', $request->phone)
             ->where('status', 'incomplete')
             ->where('id', '!=', $order->id)
             ->delete();

        // Perform Fraud Check (Async or Sync? For incomplete, maybe sync is fine as it's AJAX)
        try {
            app(FraudCheckService::class)->checkAndSave($request->phone);
        } catch (\Throwable $e) {
            // Ignore
        }

        // Sync items (delete old, add new)
        $order->items()->delete();

        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? $key;
            if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                $productId = explode('_', $key)[0];
            }

            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $productId,
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'color_id' => $item['color_id'] ?? null,
                'size_id' => $item['size_id'] ?? null,
            ]);
        }

        return response()->json(['status' => 'success', 'order_id' => $order->id]);
    }

    public function store(Request $request)
    {
        \Log::info('Checkout store method called', $request->all());

        // Adjust validation rules based on source
        $rules = [
            'phone' => 'required|string|max:20',
            'address' => 'required|string|max:255',
            'area_id' => 'required|exists:shipping_charges,id',
        ];

        if ($request->has('product_id')) {
            // Landing page / Direct checkout validation
            $rules['name'] = 'required|string|max:255';
        } else {
            // Standard checkout validation
            $rules['first_name'] = 'required|string|max:255';
            $rules['last_name'] = 'nullable|string|max:255';
            $rules['email'] = 'nullable|email|max:255';
            $rules['city'] = 'nullable|string|max:255';
            $rules['postal_code'] = 'nullable|string|max:20';
        }

        $request->validate($rules);

        try {
            DB::beginTransaction();
            \Log::info('Checkout transaction started');

            $cart = [];
            
            // Handle Direct Checkout (e.g. from Landing Page)
            if ($request->has('product_id')) {
                $product = Product::find($request->product_id);
                if (!$product) {
                    return back()->with('error', 'Product not found');
                }
                
                $price = $product->offer_price ?? $product->price;
                $variant = null;

                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $price,
                    'quantity' => $request->quantity ?? 1,
                    'color_id' => $request->filled('color_id') ? $request->color_id : null,
                    'size_id' => $request->filled('size_id') ? $request->size_id : null,
                ];
            } else {
                // Standard Cart Checkout
                $cart = Session::get('cart', []);
                if (count($cart) == 0) {
                    \Log::warning('Cart is empty during checkout');
                    return redirect()->route('shop')->with('error', 'Your cart is empty');
                }
            }

            // Validate Stock
            foreach ($cart as $key => $item) {
                $productId = $item['product_id'] ?? explode('_', $key)[0];
                $product = Product::find($productId);
                
                if (!$product) {
                    \Log::error('Product not found: ' . $item['name']);
                    return redirect()->back()->with('error', 'Product not found: ' . $item['name'])->withInput();
                }

                $quantity = $item['quantity'];
                $colorId = $item['color_id'] ?? null;
                $sizeId = $item['size_id'] ?? null;

                // Check Variant Stock
                if ($colorId || $sizeId) {
                    $variant = ProductVariant::where('product_id', $productId)
                        ->when($colorId, function($q) use ($colorId) { $q->where('color_id', $colorId); })
                        ->when($sizeId, function($q) use ($sizeId) { $q->where('size_id', $sizeId); })
                        ->first();
                    
                    if ($variant && $variant->stock_quantity < $quantity) {
                        \Log::error('Insufficient stock for variant', ['product' => $product->name]);
                        return redirect()->back()->with('error', 'Insufficient stock for ' . $product->name . ' (Variant)')->withInput();
                    }
                }

                // Check Main Product Stock
                if ($product->stock_quantity < $quantity) {
                    \Log::error('Insufficient stock for product', ['product' => $product->name]);
                    return redirect()->back()->with('error', 'Insufficient stock for ' . $product->name)->withInput();
                }
            }

            // Calculate Shipping
            $shippingCost = $this->_calculateShippingCost($request->area_id, $cart);

            $total = 0;
            foreach ($cart as $item) {
                $total += $item['price'] * $item['quantity'];
            }
            $total += $shippingCost;

            // Prevent duplicate orders (Same Phone + Same Product + Within 24 Hours)
            $checkTime = now()->subHours(24);
            $recentOrders = Order::where('phone', $request->phone)
                ->where('status', '!=', 'cancelled')
                ->where('status', '!=', 'incomplete')
                ->where('created_at', '>=', $checkTime)
                ->with('items')
                ->orderBy('created_at', 'desc')
                ->get();

            if ($recentOrders->count() > 0) {
                // Get product IDs from current request
                $currentProductIds = [];
                foreach ($cart as $key => $item) {
                    $pid = $item['product_id'] ?? explode('_', $key)[0];
                    $currentProductIds[] = (string)$pid;
                }

                foreach ($recentOrders as $recentOrder) {
                    $recentProductIds = $recentOrder->items->pluck('product_id')->map(function($id) { return (string)$id; })->toArray();
                    
                    // Check intersection
                    if (count(array_intersect($currentProductIds, $recentProductIds)) > 0) {
                        DB::rollBack();
                        
                        $timeAgo = $recentOrder->created_at->diffForHumans();
                        $contactSetting = \App\Models\ContactSetting::first();
                        $supportPhone = $contactSetting ? ($contactSetting->phone_number ?? 'our support') : 'our support';
                        
                        return redirect()->back()->with('duplicate_error', [
                            'message_en' => "You have already ordered this product {$timeAgo}. If you want to order again, please contact us.",
                            'message_bn' => "আপনি ইতিমধ্যে {$timeAgo} এই পণ্যটি অর্ডার করেছেন। আপনি যদি আবার অর্ডার করতে চান তবে অনুগ্রহ করে আমাদের সাথে যোগাযোগ করুন।",
                            'phone' => $supportPhone
                        ]);
                    }
                }
            }

            // Remove any incomplete orders (Input ID + Session based + Phone based)
            $incompleteId = $request->input('incomplete_order_id') ?? Session::get('incomplete_order_id');
            if ($incompleteId) {
                Order::where('id', $incompleteId)->delete();
                Session::forget('incomplete_order_id');
            }
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            \Log::info('Incomplete orders deleted');

            // Handle Name Splitting if needed
            $firstName = $request->first_name;
            $lastName = $request->last_name;
            if (!$firstName && $request->name) {
                $parts = explode(' ', $request->name, 2);
                $firstName = $parts[0];
                $lastName = $parts[1] ?? '-';
            }

            // Handle Defaults for Optional Fields
            $email = $request->email ?? ($request->phone . '@noemail.com');
            $city = $request->city ?? 'N/A';
            $postalCode = $request->postal_code ?? 'N/A';

            // Create Order
            $orderData = [
                'user_id' => Auth::id(), // Null if guest
                'order_number' => 'ORD-' . strtoupper(Str::random(10)),
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $email,
                'phone' => $request->phone,
                'address' => $request->address,
                'city' => $city,
                'postal_code' => $postalCode,
                'total_amount' => $total,
                'delivery_cost' => $shippingCost,
                'status' => 'pending',
                'payment_method' => $request->payment_method ?? 'cod',
                'payment_status' => 'pending',
                'source' => $request->source ?? 'website',
            ];
            \Log::info('Creating order', $orderData);
            
            $order = Order::create($orderData);
            \Log::info('Order created: ' . $order->id);

            // Send SMS Notification
            try {
                app(SmsService::class)->sendOrderConfirmation($order);
            } catch (\Throwable $e) {
                \Log::error('Failed to send order SMS: ' . $e->getMessage());
            }

            // Perform Fraud Check
            try {
                \Log::info('Starting fraud check');
                $fraudResult = app(FraudCheckService::class)->checkAndSave($request->phone);
                if (!$fraudResult['success']) {
                    \Log::error('Fraud Check Service Error: ' . ($fraudResult['message'] ?? 'Unknown error'));
                } else {
                    \Log::info('Fraud check completed successfully');
                    
                    // Explicitly update the current order to ensure it's saved in this transaction
                    $order->fraud_score = $fraudResult['percentage'];
                    $order->fraud_data = $fraudResult['data'];
                    $order->fraud_checked_at = now();
                    $order->save();
                }
            } catch (\Throwable $e) {
                // Ignore fraud check errors to not block order creation
                \Illuminate\Support\Facades\Log::error('Auto Fraud Check Failed: ' . $e->getMessage());
            }

            // Create Order Items and Decrement Stock
            foreach ($cart as $key => $item) {
                // Handle product_id (support legacy cart items where key was product_id)
                $productId = $item['product_id'] ?? $key;
                // If key is composite (e.g. 1_2_3) and product_id is missing, we need to extract it
                if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                    $productId = explode('_', $key)[0];
                }

                $quantity = $item['quantity'];
                $colorId = $item['color_id'] ?? null;
                $sizeId = $item['size_id'] ?? null;

                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $productId,
                    'quantity' => $quantity,
                    'price' => $item['price'],
                    'color_id' => $colorId,
                    'size_id' => $sizeId,
                ]);

                // Decrement Stock
                $product = Product::find($productId);
                if ($product) {
                    $product->decrement('stock_quantity', $quantity);
                    
                    if ($colorId || $sizeId) {
                        $variant = ProductVariant::where('product_id', $productId)
                            ->when($colorId, function($q) use ($colorId) { $q->where('color_id', $colorId); })
                            ->when($sizeId, function($q) use ($sizeId) { $q->where('size_id', $sizeId); })
                            ->first();
                        
                        if ($variant) {
                            $variant->decrement('stock_quantity', $quantity);
                        }
                    }
                }
            }
            \Log::info('Order items created');

            // Clear Cart (Always clear cart after successful order)
            Session::forget('cart');
            
            // Delete the incomplete order associated with this session to prevent confusion/duplication
            $incompleteId = Session::get('incomplete_order_id') ?? $request->input('incomplete_order_id');
            if ($incompleteId) {
                Order::where('id', $incompleteId)->where('status', 'incomplete')->delete();
                Session::forget('incomplete_order_id');
            }
            // Cleanup any other incomplete orders for this phone
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            
            Session::save();
            
            DB::commit();
            \Log::info('Transaction committed');

            // Fire Facebook CAPI Purchase Event
            try {
                $userData = [
                    'email' => $order->email,
                    'phone' => $order->phone,
                    'first_name' => $order->first_name,
                    'last_name' => $order->last_name,
                    'city' => $order->city,
                    'zip' => $order->postal_code,
                    'country' => 'BD',
                    'external_id' => $order->user_id ? (string)$order->user_id : null,
                ];

                $contents = [];
                foreach ($cart as $key => $item) {
                    $productId = $item['product_id'] ?? $key;
                    if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                       $productId = explode('_', $key)[0];
                   }
                   $contents[] = [
                       'id' => (string)$productId,
                       'quantity' => $item['quantity'],
                       'item_price' => $item['price'],
                       'title' => $item['name'] ?? 'Product',
                   ];
               }

                $customData = [
                    'currency' => 'BDT',
                    'value' => $order->total_amount,
                    'content_type' => 'product',
                    'contents' => $contents,
                    'order_id' => $order->order_number,
                ];

                app(FacebookConversionService::class)->track(
                    'Purchase',
                    $userData,
                    $customData,
                    $order->order_number // Event ID for Deduplication
                );
            } catch (\Exception $e) {
                \Log::error('CAPI Purchase Event Failed: ' . $e->getMessage());
            }

            // Fire TikTok CAPI Event
            try {
                $tiktokUserData = [
                    'email' => $order->email,
                    'phone' => $order->phone,
                    'external_id' => $order->user_id ? (string)$order->user_id : null,
                    'ip' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ];

                $tiktokProperties = [
                    'currency' => 'BDT',
                    'value' => $order->total_amount,
                    'content_type' => 'product',
                    'contents' => [],
                ];

                foreach ($cart as $key => $item) {
                     $productId = $item['product_id'] ?? $key;
                     if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                        $productId = explode('_', $key)[0];
                     }
                     $tiktokProperties['contents'][] = [
                        'content_id' => (string)$productId,
                        'quantity' => $item['quantity'],
                        'price' => $item['price'],
                        'content_name' => $item['name'] ?? 'Product',
                     ];
                }

                $eventName = ($order->payment_method == 'cod') ? 'CompletePayment' : 'AddPaymentInfo';
                
                app(TiktokConversionService::class)->track(
                    $eventName,
                    $tiktokUserData,
                    $tiktokProperties,
                    $order->order_number
                );
            } catch (\Exception $e) {
                \Log::error('TikTok CAPI Event Failed: ' . $e->getMessage());
            }

            // Handle Payment Redirection
            if ($order->payment_method === 'bkash') {
                return redirect()->route('payment.bkash.pay', ['order_id' => $order->id]);
            } elseif ($order->payment_method === 'sslcommerz') {
                return redirect()->route('payment.sslcz.pay', ['order_id' => $order->id]);
            }

            return redirect()->route('checkout.success', ['order' => $order->order_number]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Checkout failed: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            return redirect()->back()->with('error', 'Order failed: ' . $e->getMessage())->withInput();
        }
    }

    public function success($order)
    {
        $order = Order::where('order_number', $order)->firstOrFail();
        
        // Security check removed to allow Guest/Admin access via Order Number token
        // if (Auth::check() && $order->user_id && $order->user_id !== Auth::id()) {
        //      abort(403);
        // }

        return view('frontend.checkout.success', compact('order'));
    }
}