<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Color;
use App\Models\Size;
use App\Models\ProductVariant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use App\Services\TiktokConversionService;
use App\Services\FacebookConversionService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function index()
    {
        $cart = Session::get('cart', []);
        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        return view('frontend.cart.index', compact('cart', 'total'));
    }

    public function store(Request $request)
    {
        $product = Product::findOrFail($request->product_id);
        $cart = Session::get('cart', []);
        
        $colorId = $request->color_id;
        $sizeId = $request->size_id;
        
        // Generate unique cart key
        $cartKey = $product->id . '_' . ($colorId ?? '0') . '_' . ($sizeId ?? '0');
        
        $colorName = null;
        if ($colorId) {
            $color = Color::find($colorId);
            $colorName = $color ? $color->name : null;
        }

        $sizeName = null;
        if ($sizeId) {
            $size = Size::find($sizeId);
            $sizeName = $size ? $size->name : null;
        }
        
        // Determine Price (Product or Variant)
        $price = $product->offer_price ?? $product->price;
        $thumbnail = $product->thumbnail;
        
        if ($colorId || $sizeId) {
            $query = \App\Models\ProductVariant::where('product_id', $product->id);
            if ($colorId) $query->where('color_id', $colorId);
            if ($sizeId) $query->where('size_id', $sizeId);
            
            $variant = $query->first();
            if ($variant) {
                $price = $variant->offer_price ?? $variant->price;
                if ($variant->image) {
                    $thumbnail = $variant->image;
                }
            }
        }

        if (isset($cart[$cartKey])) {
            $cart[$cartKey]['quantity'] += $request->quantity ?? 1;
        } else {
            $cart[$cartKey] = [
                "product_id" => $product->id,
                "name" => $product->name,
                "quantity" => $request->quantity ?? 1,
                "price" => $price,
                "thumbnail" => $thumbnail,
                "slug" => $product->slug,
                "color_id" => $colorId,
                "size_id" => $sizeId,
                "color_name" => $colorName,
                "size_name" => $sizeName
            ];
        }

        Session::put('cart', $cart);

        $eventId = (string) Str::uuid();

        // Fire TikTok CAPI AddToCart Event
        try {
            $user = Auth::user();
            $tiktokUserData = [
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ];
            
            if ($user) {
                $tiktokUserData['email'] = $user->email;
                $tiktokUserData['phone'] = $user->phone;
                $tiktokUserData['external_id'] = (string)$user->id;
            }

            $tiktokProperties = [
                'currency' => 'BDT',
                'value' => $price * ($request->quantity ?? 1),
                'content_type' => 'product',
                'content_id' => (string)$product->id,
                'content_name' => $product->name,
                'quantity' => $request->quantity ?? 1,
            ];

            app(TiktokConversionService::class)->track(
                'AddToCart',
                $tiktokUserData,
                $tiktokProperties,
                $eventId
            );

            // Fire Facebook CAPI AddToCart Event
            $facebookCustomData = [
                'currency' => 'BDT',
                'value' => $price * ($request->quantity ?? 1),
                'content_type' => 'product',
                'content_ids' => [(string)$product->id],
                'content_name' => $product->name,
                'content_category' => $product->category ? $product->category->name : null,
            ];

            app(FacebookConversionService::class)->track(
                'AddToCart',
                $tiktokUserData,
                $facebookCustomData,
                $eventId
            );
            
            // Flash for Browser Pixel
            session()->flash('tiktok_event', [
                'name' => 'AddToCart',
                'params' => array_merge($tiktokProperties, ['event_id' => $eventId])
            ]);
            
            session()->flash('facebook_event', [
                'name' => 'AddToCart',
                'params' => array_merge($facebookCustomData, ['event_id' => $eventId])
            ]);

        } catch (\Exception $e) {
            \Log::error('TikTok CAPI AddToCart Event Failed: ' . $e->getMessage());
        }

        if ($request->has('buy_now')) {
            return redirect()->route('checkout.index');
        }

        return redirect()->back()->with('success', 'Product added to cart successfully!');
    }

    public function update(Request $request, $id)
    {
        if ($request->id && $request->quantity) {
            $cart = Session::get('cart');
            $cart[$request->id]["quantity"] = $request->quantity;
            Session::put('cart', $cart);
            session()->flash('success', 'Cart updated successfully');
        }
        return redirect()->back();
    }

    public function destroy($id)
    {
        if ($id) {
            $cart = Session::get('cart');
            if (isset($cart[$id])) {
                unset($cart[$id]);
                Session::put('cart', $cart);
            }
            session()->flash('success', 'Product removed successfully');
        }
        return redirect()->back();
    }
}