<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\TiktokPixel;

use App\Models\CapiLog;
use Illuminate\Support\Str;

class TiktokConversionService
{
    protected $pixels;

    public function __construct()
    {
        $this->pixels = TiktokPixel::where('is_active', true)->get();
    }

    /**
     * Track a standard event via TikTok Events API
     *
     * @param string $eventName
     * @param array $userDataArray
     * @param array $properties
     * @param string|null $eventId
     * @return void
     */
    public function track($eventName, $userDataArray = [], $properties = [], $eventId = null)
    {
        if ($this->pixels->isEmpty()) {
            return;
        }

        foreach ($this->pixels as $pixel) {
            if (empty($pixel->access_token)) {
                continue;
            }

            $payload = [];
            try {
                $payload = $this->buildPayload($eventName, $userDataArray, $properties, $eventId, $pixel);
                
                $response = Http::withHeaders([
                    'Access-Token' => $pixel->access_token,
                    'Content-Type' => 'application/json',
                ])->post('https://business-api.tiktok.com/open_api/v1.3/event/track/', $payload);

                // Log to CapiLog
                CapiLog::create([
                    'pixel_id' => $pixel->pixel_id,
                    'event_name' => $eventName,
                    'event_id' => $eventId,
                    'status' => $response->successful() ? 'success' : 'failed',
                    'response_code' => $response->status(),
                    'response_body' => Str::limit($response->body(), 1000),
                    'payload' => $payload
                ]);

                if ($response->failed()) {
                    Log::error('TikTok CAPI Error (' . $eventName . '): ' . $response->body());
                } else {
                    Log::info('TikTok CAPI Event Sent: ' . $eventName . ' | Event ID: ' . $eventId . ' | Pixel ID: ' . $pixel->pixel_id);
                }

            } catch (\Exception $e) {
                Log::error('TikTok CAPI Exception (' . $eventName . '): ' . $e->getMessage());
                
                // Log exception to DB
                CapiLog::create([
                    'pixel_id' => $pixel->pixel_id,
                    'event_name' => $eventName,
                    'event_id' => $eventId,
                    'status' => 'failed',
                    'response_code' => 500,
                    'response_body' => $e->getMessage(),
                    'payload' => $payload
                ]);
            }
        }
    }

    protected function buildPayload($eventName, $userData, $properties, $eventId, $pixel)
    {
        $userPayload = [];

        // Hash PII data (SHA-256) if provided and not already hashed
        if (isset($userData['email'])) {
            $userPayload['email'] = $this->hash($userData['email']);
        }
        if (isset($userData['phone'])) {
            $userPayload['phone_number'] = $this->hash($this->normalizePhone($userData['phone']));
        }
        
        // Non-hashed data
        if (isset($userData['ip'])) $userPayload['ip'] = $userData['ip'];
        if (isset($userData['user_agent'])) $userPayload['user_agent'] = $userData['user_agent'];
        
        // Auto-detect ttclid from session or cookie if not provided
        if (isset($userData['ttclid'])) {
            $userPayload['ttclid'] = $userData['ttclid'];
        } elseif (session()->has('ttclid')) {
            $userPayload['ttclid'] = session('ttclid');
        } elseif (request()->cookie('ttclid')) {
            $userPayload['ttclid'] = request()->cookie('ttclid');
        }

        if (isset($userData['external_id'])) $userPayload['external_id'] = $this->hash($userData['external_id']); // Usually hashed ID

        $eventData = [
            'event' => $eventName,
            'event_time' => time(),
            'event_id' => $eventId,
            'user' => $userPayload,
            'properties' => $properties,
        ];

        // Add context
        $context = [
            'page' => [
                'url' => url()->current(),
                'referrer' => request()->header('referer'),
            ],
            'user_agent' => request()->userAgent(),
            'ip' => request()->ip(),
        ];
        
        // If properties has contents, ensure it's formatted correctly
        if (isset($properties['contents'])) {
            $eventData['properties']['contents'] = $properties['contents'];
        }
        
        if (isset($properties['value'])) {
            $eventData['properties']['value'] = (float) $properties['value'];
        }
        
        if (isset($properties['currency'])) {
            $eventData['properties']['currency'] = $properties['currency'];
        }

        $payload = [
            'pixel_code' => $pixel->pixel_id,
            'event' => $eventName,
            'event_id' => $eventId,
            'timestamp' => gmdate('Y-m-d\TH:i:s\Z'), // ISO 8601
            'context' => $context,
            'properties' => $eventData['properties'], // TikTok expects properties at top level in some docs, but actually inside 'event' object usually? 
            // Wait, looking at docs: 
            // POST https://business-api.tiktok.com/open_api/v1.3/event/track/
            // Body: { "pixel_code": "...", "event": "Purchase", "event_id": "...", "timestamp": "...", "context": {...}, "properties": {...} }
        ];

        // Re-structuring based on official docs for Single Event
        // Actually, the structure is:
        /*
        {
            "pixel_code": "<pixel_code>",
            "event": "PlaceAnOrder",
            "event_id": "1616318632825_357",
            "timestamp": "2020-09-17T19:49:27+08:00",
            "context": {
                "ad": {
                    "callback": "123" // ttclid
                },
                "page": {
                    "url": "http://demo.mywebsite.com/purchase",
                    "referrer": "http://demo.mywebsite.com"
                },
                "user": {
                    "external_id": "f0e388f5390e75b7...",
                    "phone_number": "2f9d2b4e3e2e2a...",
                    "email": "3525e11f81..."
                },
                "ip": "13.57.97.131",
                "user_agent": "Mozilla/5.0..."
            },
            "properties": {
                "contents": [
                    {
                        "content_id": "301",
                        "content_name": "Dress",
                        "quantity": 1,
                        "price": 398
                    }
                ],
                "content_type": "product",
                "value": 398,
                "currency": "USD"
            }
        }
        */

        // Let's adjust to match the structure above
        $finalPayload = [
            'pixel_code' => $pixel->pixel_id,
            'event' => $eventName,
            'event_id' => $eventId,
            'timestamp' => date('c'), // ISO 8601
            'context' => [
                'page' => [
                    'url' => url()->current(),
                    'referrer' => request()->header('referer'),
                ],
                'user' => $userPayload,
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ],
            'properties' => $properties,
        ];

        if (isset($userData['ttclid'])) {
            $finalPayload['context']['ad']['callback'] = $userData['ttclid'];
        }

        if ($pixel->test_event_code) {
            $finalPayload['test_event_code'] = $pixel->test_event_code;
        }

        return $finalPayload;
    }

    private function hash($value)
    {
        if (empty($value)) return null;
        // Check if already hashed (simple check for sha256 length)
        if (preg_match('/^[a-f0-9]{64}$/', $value)) {
            return $value;
        }
        return hash('sha256', trim(strtolower($value)));
    }

    /**
     * Normalize Phone Number
     */
    private function normalizePhone($phone)
    {
        // Remove non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // If it starts with '01' (standard BD mobile prefix) and is 11 digits long, prepend '88'
        if (strlen($phone) === 11 && substr($phone, 0, 2) === '01') {
            $phone = '88' . $phone;
        }
        
        return $phone;
    }
}
