<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use App\Models\GeneralSetting;
use App\Models\Order;

class SteadfastService
{
    protected $baseUrl = 'https://portal.packzy.com/api/v1';
    protected $apiKey;
    protected $secretKey;

    public function __construct()
    {
        $setting = GeneralSetting::first();
        if ($setting) {
            $this->apiKey = $setting->courier_steadfast_api_key;
            $this->secretKey = $setting->courier_steadfast_secret_key;
        }
    }

    private function getHeaders()
    {
        return [
            'Api-Key' => $this->apiKey,
            'Secret-Key' => $this->secretKey,
            'Content-Type' => 'application/json'
        ];
    }

    public function getBalance()
    {
        if (!$this->apiKey || !$this->secretKey) {
            return null;
        }

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->get($this->baseUrl . '/get_balance');

            if ($response->successful()) {
                return $response->json()['current_balance'] ?? null;
            }
        } catch (\Exception $e) {
            return null;
        }

        return null;
    }

    public function createOrder(Order $order)
    {
        if (!$this->apiKey || !$this->secretKey) {
            return ['status' => 'error', 'message' => 'API Credentials not configured'];
        }

        // Prepare payload
        $payload = [
            'invoice' => $order->order_number,
            'recipient_name' => $order->first_name . ' ' . $order->last_name,
            'recipient_phone' => $order->phone,
            'recipient_address' => $order->address,
            'cod_amount' => $order->payment_status == 'paid' ? 0 : $order->total_amount,
            'note' => $order->note,
        ];

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->post($this->baseUrl . '/create_order', $payload);

            $data = $response->json();

            if ($response->successful() && isset($data['consignment']['consignment_id'])) {
                return [
                    'status' => 'success',
                    'consignment_id' => $data['consignment']['consignment_id'],
                    'tracking_code' => $data['consignment']['tracking_code'],
                    'courier_status' => $data['consignment']['status'] ?? 'pending',
                    'data' => $data
                ];
            } else {
                return [
                    'status' => 'error', 
                    'message' => $data['message'] ?? 'Unknown API Error',
                    'data' => $data
                ];
            }
        } catch (\Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    public function checkStatus($consignmentId)
    {
        if (!$this->apiKey || !$this->secretKey) {
            return ['status' => 'error', 'message' => 'API Credentials not configured'];
        }

        try {
            $response = Http::withHeaders($this->getHeaders())
                ->get($this->baseUrl . '/status_by_cid/' . $consignmentId);

            if ($response->successful()) {
                return $response->json();
            }
        } catch (\Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }

        return ['status' => 'error', 'message' => 'Failed to fetch status'];
    }
}
