<?php

namespace App\Services;

use App\Models\Order;
use App\Models\GeneralSetting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FraudCheckService
{
    /**
     * Check fraud status for a phone number and save result to orders.
     *
     * @param string $phone
     * @return array
     */
    public function checkAndSave($phone)
    {
        // 1. Get API Key
        $settings = GeneralSetting::first();
        if (!$settings) {
             return [
                'success' => false,
                'message' => 'General Settings not found'
            ];
        }
        $apiKey = $settings->fraud_api_key ?? null;

        if (!$apiKey) {
            return [
                'success' => false,
                'message' => 'API Key not configured'
            ];
        }

        // 2. Call External API
        try {
            $response = Http::asForm()->withHeaders([
                'Authorization' => 'Bearer ' . $apiKey
            ])->post('https://fraudchecker.link/api/v1/qc/', [
                'phone' => $phone
            ]);

            if ($response->successful()) {
                $data = $response->json();

                // 3. Calculate Score
                $total = intval($data['total_parcels'] ?? 0);
                $delivered = intval($data['total_delivered'] ?? 0);
                $percentage = $total > 0 ? round(($delivered / $total) * 100) : 0;

                // 4. Save to Database (Update all orders with this phone)
                // We only update if fraud_score is null OR if we want to refresh?
                // The user said "once checked it should be saved". 
                // Updating all ensures consistency.
                Order::where('phone', $phone)->update([
                    'fraud_score' => $percentage,
                    'fraud_data' => json_encode($data), // Store as JSON string if column is text, or array if cast handled? 
                    // Laravel casts: if cast is 'array', we pass array. update() bypasses mutators? 
                    // Eloquent update() on query builder DOES NOT trigger mutators/casts automatically for JSON usually unless using model instance.
                    // But here we are doing mass update.
                    // If column is JSON type in DB, we can pass array if using Model::create/update with cast.
                    // But Order::where(...)->update(...) is a Query Builder update. It expects raw values.
                    // So we should json_encode it.
                    'fraud_checked_at' => now(),
                ]);
                
                // Note: 'fraud_data' column is json/longtext. Query builder needs string.
                // Wait, if I use json_encode here, and the column is JSON type, it works.

                return [
                    'success' => true,
                    'data' => $data,
                    'percentage' => $percentage
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'API Error: ' . $response->body()
                ];
            }

        } catch (\Exception $e) {
            Log::error('Fraud check failed: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
