<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Cookie;
use App\Models\GeneralSetting;
use App\Models\FacebookPixel;
use App\Models\CapiLog;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;

class FacebookConversionService
{
    /**
     * Track a standard event using Direct HTTP Request (CAPI)
     *
     * @param string $eventName
     * @param array $userDataArray
     * @param array $customDataArray
     * @param string|null $eventId
     * @return void
     */
    public function track($eventName, $userDataArray = [], $customDataArray = [], $eventId = null)
    {
        // 1. Retrieve Active Pixels
        $pixels = collect();

        try {
            if (Schema::hasTable('facebook_pixels')) {
                $pixels = FacebookPixel::where('is_active', true)
                    ->where('server_side_enabled', true)
                    ->get();
            }
        } catch (\Exception $e) {
            Log::error('FacebookPixel Table Error: ' . $e->getMessage());
        }

        // Fallback to GeneralSetting if no pixels in dedicated table
        if ($pixels->isEmpty()) {
            try {
                $setting = GeneralSetting::first();
                if ($setting && $setting->facebook_pixel_enabled && $setting->facebook_pixel_id && $setting->facebook_access_token) {
                    $pixels->push((object)[
                        'pixel_id' => $setting->facebook_pixel_id,
                        'access_token' => $setting->facebook_access_token,
                        'test_event_code' => $setting->facebook_test_event_code ?? null,
                        'api_version' => 'v19.0' // Default fallback
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('GeneralSetting Table Error: ' . $e->getMessage());
            }
        }

        if ($pixels->isEmpty()) {
            return;
        }

        // 2. Prepare User Data (Common for all pixels)
        $userData = $this->prepareUserData($userDataArray);

        // 3. Prepare Custom Data (Common for all pixels)
        $customData = $this->prepareCustomData($customDataArray);

        // 4. Send Event for each Pixel
        foreach ($pixels as $pixel) {
            $this->sendEventToPixel($pixel, $eventName, $userData, $customData, $eventId);
        }
    }

    /**
     * Prepare and Hash User Data
     */
    private function prepareUserData($data)
    {
        $userData = [
            'client_ip_address' => $data['ip'] ?? Request::ip(),
            'client_user_agent' => $data['user_agent'] ?? Request::userAgent(),
        ];

        // Hashing PII
        if (!empty($data['email'])) {
            $userData['em'] = hash('sha256', strtolower(trim($data['email'])));
        }
        if (!empty($data['phone'])) {
            $userData['ph'] = hash('sha256', $this->normalizePhone($data['phone']));
        }
        if (!empty($data['first_name'])) {
            $userData['fn'] = hash('sha256', strtolower(trim($data['first_name'])));
        }
        if (!empty($data['last_name'])) {
            $userData['ln'] = hash('sha256', strtolower(trim($data['last_name'])));
        }
        if (!empty($data['city'])) {
            $userData['ct'] = hash('sha256', strtolower(trim($data['city'])));
        }
        if (!empty($data['state'])) {
            $userData['st'] = hash('sha256', strtolower(trim($data['state'])));
        }
        if (!empty($data['zip'])) {
            $userData['zp'] = hash('sha256', strtolower(trim($data['zip'])));
        }
        if (!empty($data['country'])) {
            $userData['country'] = hash('sha256', strtolower(trim($data['country'])));
        }
        if (!empty($data['external_id'])) {
            $userData['external_id'] = hash('sha256', (string)trim($data['external_id']));
        }

        // Cookies (fbp, fbc)
        $fbp = Cookie::get('_fbp') ?? Request::cookie('_fbp') ?? null;
        $fbc = Cookie::get('_fbc') ?? Request::cookie('_fbc') ?? null;

        if ($fbp) {
            $userData['fbp'] = $fbp;
        }
        if ($fbc) {
            $userData['fbc'] = $fbc;
        }

        return array_filter($userData);
    }

    /**
     * Normalize Phone Number
     */
    private function normalizePhone($phone)
    {
        // Remove non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // If it starts with '01' (standard BD mobile prefix) and is 11 digits long, prepend '88'
        if (strlen($phone) === 11 && substr($phone, 0, 2) === '01') {
            $phone = '88' . $phone;
        }
        
        // If it starts with '880', ensure it's kept as is (already correct format)
        
        return $phone;
    }

    /**
     * Prepare Custom Data
     */
    private function prepareCustomData($data)
    {
        $customData = [];

        if (isset($data['currency'])) $customData['currency'] = $data['currency'];
        if (isset($data['value'])) $customData['value'] = $data['value'];
        if (isset($data['content_ids'])) $customData['content_ids'] = $data['content_ids'];
        if (isset($data['content_type'])) $customData['content_type'] = $data['content_type'];
        if (isset($data['content_name'])) $customData['content_name'] = $data['content_name'];
        if (isset($data['content_category'])) $customData['content_category'] = $data['content_category'];
        if (isset($data['order_id'])) $customData['order_id'] = $data['order_id'];
        if (isset($data['num_items'])) $customData['num_items'] = $data['num_items'];
        if (isset($data['search_string'])) $customData['search_string'] = $data['search_string'];
        if (isset($data['status'])) $customData['status'] = $data['status'];
        
        if (isset($data['contents']) && is_array($data['contents'])) {
            $contents = [];
            $generatedContentIds = [];
            
            foreach ($data['contents'] as $item) {
                $itemId = $item['id'] ?? null;
                $contentItem = [
                    'id' => $itemId,
                    'quantity' => $item['quantity'] ?? 1,
                    'item_price' => $item['item_price'] ?? null
                ];
                if(isset($item['title'])) $contentItem['title'] = $item['title'];
                $contents[] = $contentItem;

                if ($itemId) {
                    $generatedContentIds[] = (string)$itemId;
                }
            }
            $customData['contents'] = $contents;

            // Auto-fill content_ids if missing
            if (!isset($customData['content_ids']) && !empty($generatedContentIds)) {
                $customData['content_ids'] = $generatedContentIds;
            }

            // Auto-fill num_items if missing
            if (!isset($customData['num_items'])) {
                $customData['num_items'] = count($contents);
            }
        }

        return $customData;
    }

    /**
     * Send Event to Facebook Graph API
     */
    private function sendEventToPixel($pixel, $eventName, $userData, $customData, $eventId)
    {
        if (empty($pixel->pixel_id) || empty($pixel->access_token)) {
            return;
        }

        // Use pixel-specific API version or default
        $apiVersion = $pixel->api_version ?? 'v19.0';
        $url = "https://graph.facebook.com/{$apiVersion}/{$pixel->pixel_id}/events";

        // Prepare Event Data
        $eventData = [
            'event_name' => $eventName,
            'event_time' => time(),
            'event_id' => $eventId,
            'event_source_url' => request()->header('referer') ?? request()->fullUrl(),
            'user_data' => $userData,
            'custom_data' => $customData,
            'action_source' => 'website',
        ];

        // Prepare Payload
        $payload = [
            'data' => [$eventData],
        ];

        // Add Test Event Code (outside data array)
        if (!empty($pixel->test_event_code)) {
            $payload['test_event_code'] = $pixel->test_event_code;
        }

        // Access Token is passed as query parameter
        $urlWithToken = $url . '?access_token=' . $pixel->access_token;

        try {
            $response = Http::post($urlWithToken, $payload);

            // Log to CapiLog
            CapiLog::create([
                'pixel_id' => $pixel->pixel_id,
                'event_name' => $eventName,
                'event_id' => $eventId,
                'status' => $response->successful() ? 'success' : 'failed',
                'response_code' => $response->status(),
                'response_body' => Str::limit($response->body(), 1000),
                'payload' => $payload
            ]);

            if ($response->failed()) {
                Log::error("Facebook CAPI Error ({$eventName}): " . $response->body());
            }

        } catch (\Exception $e) {
            Log::error("Facebook CAPI Exception ({$eventName}): " . $e->getMessage());
            
            CapiLog::create([
                'pixel_id' => $pixel->pixel_id,
                'event_name' => $eventName,
                'event_id' => $eventId,
                'status' => 'failed',
                'response_code' => 500,
                'response_body' => $e->getMessage(),
                'payload' => $payload
            ]);
        }
    }
}
