<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Response;
use App\Models\Order;
use App\Models\OrderCall;
use App\Models\GeneralSetting;
use Illuminate\Support\Facades\Http;

class VoiceController extends Controller
{
    public function start(Request $request, Order $order)
    {
        $order->update(['call_status' => 'ANSWERED_WAITING_INPUT']);

        $message = "This is an automated order verification from Rayhan Shop. "
                 . "Press 1 to confirm your order. "
                 . "Press 2 to talk to our agent.";

        $gatherAction = route('voice.input', ['order' => $order->id]);

        $twiml = '<?xml version="1.0" encoding="UTF-8"?>'
            . '<Response>'
            . '<Say voice="alice">' . htmlspecialchars($message) . '</Say>'
            . '<Pause length="1"/>'
            . '<Gather input="dtmf" numDigits="1" timeout="5" action="' . htmlspecialchars($gatherAction) . '" method="POST">'
            . '<Say voice="alice">Press 1 to confirm. Press 2 to talk to our agent.</Say>'
            . '</Gather>'
            . '<Say voice="alice">No input received. Goodbye.</Say>'
            . '</Response>';

        return Response::make($twiml, 200, ['Content-Type' => 'text/xml']);
    }

    public function input(Request $request, Order $order)
    {
        $digit = $request->input('Digits');
        $callSid = $request->input('CallSid');

        if ($callSid) {
            OrderCall::firstOrCreate(
                ['call_sid' => $callSid],
                ['order_id' => $order->id, 'attempt' => 1]
            )->update(['dtmf_input' => $digit]);
        }

        if ($digit === '1') {
            $order->update(['call_status' => 'CONFIRMED_BY_DTMF_1', 'status' => 'confirmed']);
            $twiml = '<?xml version="1.0" encoding="UTF-8"?>'
                . '<Response>'
                . '<Say voice="alice">Thank you. Your order is confirmed.</Say>'
                . '</Response>';
        } elseif ($digit === '2') {
            $order->update(['call_status' => 'TRANSFERRED_TO_AGENT']);
            $setting = GeneralSetting::first();
            $agentNumber = $setting->call_center_number ?? null;

            if ($agentNumber) {
                $twiml = '<?xml version="1.0" encoding="UTF-8"?>'
                    . '<Response>'
                    . '<Say voice="alice">Connecting you to our agent.</Say>'
                    . '<Dial>' . htmlspecialchars($agentNumber) . '</Dial>'
                    . '</Response>';
            } else {
                $twiml = '<?xml version="1.0" encoding="UTF-8"?>'
                    . '<Response>'
                    . '<Say voice="alice">Sorry, agent number is not configured.</Say>'
                    . '</Response>';
            }
        } else {
            $order->update(['call_status' => 'NO_INPUT']);
            $twiml = '<?xml version="1.0" encoding="UTF-8"?>'
                . '<Response>'
                . '<Say voice="alice">No valid input received. Goodbye.</Say>'
                . '</Response>';
        }

        return Response::make($twiml, 200, ['Content-Type' => 'text/xml']);
    }

    public function status(Request $request)
    {
        $orderId = $request->query('order_id') ?? $request->input('order_id');
        $callSid = $request->input('CallSid');
        $status = $request->input('CallStatus'); // queued, ringing, in-progress, completed, busy, failed, no-answer, canceled
        $duration = $request->input('CallDuration');
        $answeredBy = $request->input('AnsweredBy');

        Log::info('Twilio status callback', [
            'order_id' => $orderId,
            'call_sid' => $callSid,
            'status' => $status,
            'duration' => $duration,
            'answered_by' => $answeredBy,
        ]);

        if ($orderId) {
            $order = Order::find($orderId);
            if ($order) {
                $map = [
                    'queued' => 'CALL_QUEUED',
                    'ringing' => 'CALL_INITIATED',
                    'in-progress' => 'ANSWERED_WAITING_INPUT',
                    'completed' => $order->call_status ?? 'NO_INPUT',
                    'no-answer' => 'NOT_ANSWERED',
                    'busy' => 'NOT_ANSWERED',
                    'failed' => 'FAILED',
                    'canceled' => 'FAILED',
                ];
                $order->update(['call_status' => $map[$status] ?? $order->call_status]);
            }
        }

        if ($callSid) {
            $log = OrderCall::firstOrCreate(['call_sid' => $callSid], [
                'order_id' => $orderId,
                'attempt' => 1,
            ]);
            $log->update([
                'duration' => $duration,
                'call_result' => $status,
            ]);
        }

        return response()->json(['ok' => true]);
    }
}

