<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;
use App\Services\TiktokConversionService;
use App\Services\FacebookConversionService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('status', true);

        // Filter by Category
        if ($request->has('category')) {
            $slug = $request->get('category');
            $category = Category::where('slug', $slug)->first();
            if ($category) {
                // Include products from subcategories
                $categoryIds = $category->children()->pluck('id')->push($category->id);
                $query->whereIn('category_id', $categoryIds);
            }
        }

        // Filter by Brand
        if ($request->has('brand')) {
            $slug = $request->get('brand');
            $brand = Brand::where('slug', $slug)->first();
            if ($brand) {
                $query->where('brand_id', $brand->id);
            }
        }

        // Filter by Availability
        if ($request->has('availability')) {
            $availability = (array) $request->get('availability');
            if (in_array('in_stock', $availability) && !in_array('out_of_stock', $availability)) {
                $query->where('stock_quantity', '>', 0);
            } elseif (in_array('out_of_stock', $availability) && !in_array('in_stock', $availability)) {
                $query->where('stock_quantity', '<=', 0);
            }
        }

        // Filter by Price Range
        if ($request->has('min_price') && $request->get('min_price') != '') {
            $query->where('price', '>=', $request->get('min_price'));
        }
        if ($request->has('max_price') && $request->get('max_price') != '') {
            $query->where('price', '<=', $request->get('max_price'));
        }

        // Search
        $searchEventId = null;
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where('name', 'like', "%{$search}%");

            // Fire Search Event (CAPI)
            try {
                $searchEventId = (string) Str::uuid();
                $user = Auth::user();
                $userData = [
                    'ip' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ];
                if ($user) {
                    $userData['email'] = $user->email;
                    $userData['phone'] = $user->phone;
                    $userData['external_id'] = (string)$user->id;
                }

                $customData = [
                    'search_string' => $search,
                    'content_type' => 'product',
                ];

                app(FacebookConversionService::class)->track(
                    'Search',
                    $userData,
                    $customData,
                    $searchEventId
                );

                // TikTok Search Event
                app(TiktokConversionService::class)->track(
                    'Search',
                    $userData,
                    ['query' => $search],
                    $searchEventId
                );

            } catch (\Exception $e) {
                \Log::error('CAPI Search Event Failed: ' . $e->getMessage());
            }
        }

        // Sorting
        if ($request->has('sort')) {
            switch ($request->get('sort')) {
                case 'price_asc':
                    $query->orderBy('price', 'asc');
                    break;
                case 'price_desc':
                    $query->orderBy('price', 'desc');
                    break;
                case 'newest':
                    $query->latest();
                    break;
                default:
                    $query->latest();
            }
        } else {
            $query->latest();
        }

        $products = $query->paginate(12)->withQueryString();
        $categories = Category::whereNull('parent_id')->with('children')->get();
        $brands = Brand::where('status', true)->get();
        
        $minPrice = Product::min('price') ?? 0;
        $maxPrice = Product::max('price') ?? 0;

        return view('frontend.shop.index', compact('products', 'categories', 'brands', 'minPrice', 'maxPrice', 'searchEventId'));
    }

    public function show($slug)
    {
        $product = Product::where('slug', $slug)
            ->where('status', true)
            ->with(['category', 'brand', 'colors', 'sizes', 'images', 'reviews.user', 'variants.color', 'variants.size'])
            ->firstOrFail();

        $relatedProducts = Product::where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->where('status', true)
            ->take(4)
            ->get();

        $eventId = (string) Str::uuid();

        // Fire Facebook & TikTok CAPI ViewContent Event
        try {
            $user = Auth::user();
            $userData = [
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'fbc' => request()->cookie('_fbc'),
                'fbp' => request()->cookie('_fbp'),
            ];
            
            if ($user) {
                $userData['email'] = $user->email;
                $userData['phone'] = $user->phone;
                $userData['external_id'] = (string)$user->id;
            }

            // TikTok specific data
            $tiktokProperties = [
                'currency' => 'BDT',
                'value' => $product->offer_price ?? $product->price,
                'content_type' => 'product',
                'content_id' => (string)$product->id,
                'content_name' => $product->name,
                'content_category' => $product->category ? $product->category->name : null,
            ];

            // Facebook specific data
            $facebookCustomData = [
                'currency' => 'BDT',
                'value' => $product->offer_price ?? $product->price,
                'content_type' => 'product',
                'content_ids' => [(string)$product->id],
                'content_name' => $product->name,
                'content_category' => $product->category ? $product->category->name : null,
            ];

            // Fire TikTok Event
            app(TiktokConversionService::class)->track(
                'ViewContent',
                $userData,
                $tiktokProperties,
                $eventId
            );

            // Fire Facebook Event
            app(FacebookConversionService::class)->track(
                'ViewContent',
                $userData,
                $facebookCustomData,
                $eventId
            );

        } catch (\Exception $e) {
            \Log::error('CAPI ViewContent Event Failed: ' . $e->getMessage());
        }

        return view('frontend.shop.show', compact('product', 'relatedProducts', 'eventId'));
    }

    public function searchSuggestions(Request $request)
    {
        $query = $request->get('query');
        
        if (strlen($query) < 1) {
            return response()->json([]);
        }

        $products = Product::where('status', true)
            ->where('name', 'like', "%{$query}%")
            ->select('id', 'name', 'slug', 'thumbnail', 'price', 'offer_price')
            ->take(8)
            ->get()
            ->map(function ($p) {
                $thumb = $p->thumbnail ?? '';
                if ($thumb) {
                    if (filter_var($thumb, FILTER_VALIDATE_URL)) {
                        $thumbUrl = $thumb;
                    } else {
                        $thumbUrl = asset('storage/' . ltrim($thumb, '/'));
                    }
                } else {
                    $thumbUrl = asset('images/no-image.svg');
                }

                return [
                    'id' => $p->id,
                    'name' => $p->name,
                    'slug' => $p->slug,
                    'thumbnail_url' => $thumbUrl,
                    'price' => $p->offer_price && $p->offer_price > 0 ? $p->offer_price : $p->price,
                    'original_price' => $p->price,
                ];
            });
        
        return response()->json($products);
    }
}
