<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\GeneralSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use App\Services\TiktokConversionService;

class PaymentController extends Controller
{
    // bKash Configuration
    private $bkash_base_url;
    private $bkash_app_key;
    private $bkash_app_secret;
    private $bkash_username;
    private $bkash_password;

    // SSLCommerz Configuration
    private $sslcz_base_url;
    private $sslcz_store_id;
    private $sslcz_store_password;

    public function __construct()
    {
        $settings = GeneralSetting::first();
        if ($settings) {
            // bKash Config
            $this->bkash_base_url = $settings->bkash_test_mode 
                ? 'https://tokenized.sandbox.bka.sh/v1.2.0-beta' 
                : 'https://tokenized.pay.bka.sh/v1.2.0-beta';
            $this->bkash_app_key = $settings->bkash_app_key;
            $this->bkash_app_secret = $settings->bkash_app_secret;
            $this->bkash_username = $settings->bkash_username;
            $this->bkash_password = $settings->bkash_password;

            // SSLCommerz Config
            $this->sslcz_base_url = $settings->sslcz_test_mode 
                ? 'https://sandbox.sslcommerz.com/gwprocess/v4/api.php' 
                : 'https://securepay.sslcommerz.com/gwprocess/v4/api.php';
            $this->sslcz_store_id = $settings->sslcz_store_id;
            $this->sslcz_store_password = $settings->sslcz_store_password;
        }
    }

    // ==========================================
    // bKash Payment Methods
    // ==========================================

    private function getBkashToken()
    {
        try {
            $response = Http::timeout(30)->withHeaders([
                'Content-Type' => 'application/json',
                'username' => $this->bkash_username,
                'password' => $this->bkash_password,
            ])->post($this->bkash_base_url . '/tokenized/checkout/token/grant', [
                'app_key' => $this->bkash_app_key,
                'app_secret' => $this->bkash_app_secret,
            ]);

            if ($response->successful()) {
                return $response->json()['id_token'];
            }

            Log::error('bKash Token Error: ' . $response->body());
            return null;
        } catch (\Exception $e) {
            Log::error('bKash Token Exception: ' . $e->getMessage());
            return null;
        }
    }

    public function bkashPay($order_id)
    {
        $order = Order::findOrFail($order_id);
        
        $token = $this->getBkashToken();
        if (!$token) {
            return redirect()->route('checkout.success', ['order' => $order->order_number])
                ->with('error', 'Unable to connect to bKash. Please check your internet connection or try again later.');
        }

        try {
            $request_data = [
                'mode' => '0011',
                'payerReference' => $order->order_number,
                'callbackURL' => route('payment.bkash.callback', ['order_id' => $order->id]),
                'amount' => number_format($order->total_amount, 2, '.', ''),
                'currency' => 'BDT',
                'intent' => 'sale',
                'merchantInvoiceNumber' => $order->order_number
            ];

            $response = Http::timeout(30)->withHeaders([
                'Content-Type' => 'application/json',
                'Authorization' => $token,
                'X-APP-Key' => $this->bkash_app_key,
            ])->post($this->bkash_base_url . '/tokenized/checkout/create', $request_data);

            $data = $response->json();

            if (isset($data['bkashURL'])) {
                // Store token for callback
                Session::put('bkash_token', $token);
                return redirect($data['bkashURL']);
            }

            Log::error('bKash Create Payment Error: ' . $response->body());
            return redirect()->route('checkout.success', ['order' => $order->order_number])
                ->with('error', 'Failed to initiate bKash payment: ' . ($data['statusMessage'] ?? 'Unknown Error'));

        } catch (\Exception $e) {
            Log::error('bKash Create Payment Exception: ' . $e->getMessage());
            return redirect()->route('checkout.success', ['order' => $order->order_number])
                ->with('error', 'Payment initiation failed due to a technical error. Please try again.');
        }
    }

    public function bkashCallback(Request $request)
    {
        $status = $request->input('status');
        $paymentId = $request->input('paymentID');
        $orderId = $request->input('order_id'); // Retrieved from query param
        
        // Helper to find order
        $order = null;
        if ($orderId) {
            $order = Order::find($orderId);
        }

        if ($status === 'success') {
            $token = Session::get('bkash_token');
            if (!$token) {
                $token = $this->getBkashToken();
            }

            try {
                $response = Http::timeout(30)->withHeaders([
                    'Content-Type' => 'application/json',
                    'Authorization' => $token,
                    'X-APP-Key' => $this->bkash_app_key,
                ])->post($this->bkash_base_url . '/tokenized/checkout/execute', [
                    'paymentID' => $paymentId
                ]);

                $data = $response->json();

                if (isset($data['statusCode']) && $data['statusCode'] === '0000') {
                    // Payment Success
                    $orderNumber = $data['merchantInvoiceNumber'];
                    $trxId = $data['trxID'];
                    
                    // Re-fetch order if needed (though we might have it from ID)
                    if (!$order) {
                        $order = Order::where('order_number', $orderNumber)->first();
                    }

                    if ($order) {
                        $order->payment_status = 'paid';
                        $order->transaction_id = $trxId;
                        $order->payment_details = json_encode($data);
                        $order->save();

                        // Fire TikTok CAPI CompletePayment
                        try {
                            $tiktokUserData = [
                                'email' => $order->email,
                                'phone' => $order->phone,
                                'external_id' => $order->user_id ? (string)$order->user_id : null,
                                'ip' => request()->ip(),
                                'user_agent' => request()->userAgent(),
                            ];
            
                            $tiktokProperties = [
                                'currency' => 'BDT',
                                'value' => $order->total_amount,
                                'content_type' => 'product',
                                'contents' => [],
                            ];
                            
                            foreach ($order->items as $item) {
                                 $tiktokProperties['contents'][] = [
                                    'content_id' => (string)$item->product_id,
                                    'quantity' => $item->quantity,
                                    'price' => $item->price,
                                    'content_name' => 'Product', // Keeping it simple to avoid N+1 query or just 'Product'
                                 ];
                            }
            
                            app(TiktokConversionService::class)->track(
                                'CompletePayment',
                                $tiktokUserData,
                                $tiktokProperties,
                                $order->order_number
                            );
                        } catch (\Exception $e) {
                            \Log::error('TikTok CAPI CompletePayment Failed (bKash): ' . $e->getMessage());
                        }
                        
                        Session::forget('bkash_token');
                        return redirect()->route('checkout.success', ['order' => $order->order_number])->with('success', 'Payment Successful via bKash');
                    }
                } else {
                    Log::error('bKash Execute Error: ' . json_encode($data));
                    
                    if ($order) {
                        return redirect()->route('checkout.success', ['order' => $order->order_number])
                            ->with('error', 'bKash Payment Failed: ' . ($data['statusMessage'] ?? 'Execution Failed'));
                    }
                }
            } catch (\Exception $e) {
                Log::error('bKash Execute Exception: ' . $e->getMessage());
            }
        }

        // Failure/Cancel scenarios
        if ($order) {
            return redirect()->route('checkout.success', ['order' => $order->order_number])
                ->with('error', 'bKash Payment ' . ucfirst($status));
        }
        
        return redirect()->route('shop')->with('error', 'bKash Payment Cancelled or Failed.');
    }

    // ==========================================
    // SSLCommerz Payment Methods
    // ==========================================

    public function sslczPay($order_id)
    {
        $order = Order::findOrFail($order_id);

        try {
            $post_data = [];
            $post_data['store_id'] = $this->sslcz_store_id;
            $post_data['store_passwd'] = $this->sslcz_store_password;
            $post_data['total_amount'] = $order->total_amount;
            $post_data['currency'] = "BDT";
            $post_data['tran_id'] = $order->order_number;
            $post_data['success_url'] = route('payment.sslcz.callback', ['order_id' => $order->id]);
            $post_data['fail_url'] = route('payment.sslcz.callback', ['order_id' => $order->id]);
            $post_data['cancel_url'] = route('payment.sslcz.callback', ['order_id' => $order->id]);
            
            // Customer Info
            $post_data['cus_name'] = $order->first_name . ' ' . $order->last_name;
            $post_data['cus_email'] = $order->email;
            $post_data['cus_add1'] = $order->address;
            $post_data['cus_city'] = $order->city;
            $post_data['cus_postcode'] = $order->postal_code;
            $post_data['cus_country'] = "Bangladesh";
            $post_data['cus_phone'] = $order->phone;

            // Shipping Info (Optional but good practice)
            $post_data['shipping_method'] = "NO";
            $post_data['product_name'] = "Order " . $order->order_number;
            $post_data['product_category'] = "Goods";
            $post_data['product_profile'] = "general";

            $response = Http::timeout(30)->asForm()->post($this->sslcz_base_url, $post_data);
            
            $sslcz = $response->json();

            if (isset($sslcz['status']) && $sslcz['status'] == 'SUCCESS') {
                return redirect($sslcz['GatewayPageURL']);
            } else {
                Log::error('SSLCommerz Init Error: ' . $response->body());
                return redirect()->route('checkout.success', ['order' => $order->order_number])
                    ->with('error', 'Failed to connect to SSLCommerz: ' . ($sslcz['failedreason'] ?? 'Unknown Error'));
            }
        } catch (\Exception $e) {
            Log::error('SSLCommerz Init Exception: ' . $e->getMessage());
            return redirect()->route('checkout.success', ['order' => $order->order_number])
                ->with('error', 'Payment initiation failed due to a technical error. Please try again.');
        }
    }

    public function sslczCallback(Request $request)
    {
        $tran_id = $request->input('tran_id');
        $val_id = $request->input('val_id');
        $amount = $request->input('amount');
        $currency = $request->input('currency_type');
        $status = $request->input('status');
        $orderId = $request->input('order_id');

        $order = null;
        if ($orderId) {
            $order = Order::find($orderId);
        }
        
        if (!$order && $tran_id) {
            $order = Order::where('order_number', $tran_id)->first();
        }

        if (!$order) {
            return redirect()->route('shop')->with('error', 'Invalid Transaction ID');
        }

        if ($status == 'VALID' || $status == 'VALIDATED') {
            // Validate amount
            if ($currency == "BDT" && $amount >= $order->total_amount) {
                $order->payment_status = 'paid';
                $order->transaction_id = $val_id; // Using val_id as reliable ref
                $order->payment_details = json_encode($request->all());
                $order->save();

                // Fire TikTok CAPI CompletePayment
                try {
                    $tiktokUserData = [
                        'email' => $order->email,
                        'phone' => $order->phone,
                        'external_id' => $order->user_id ? (string)$order->user_id : null,
                        'ip' => request()->ip(),
                        'user_agent' => request()->userAgent(),
                    ];
    
                    $tiktokProperties = [
                        'currency' => 'BDT',
                        'value' => $order->total_amount,
                        'content_type' => 'product',
                        'contents' => [],
                    ];
                    
                    foreach ($order->items as $item) {
                            $tiktokProperties['contents'][] = [
                            'content_id' => (string)$item->product_id,
                            'quantity' => $item->quantity,
                            'price' => $item->price,
                            'content_name' => 'Product',
                            ];
                    }
    
                    app(TiktokConversionService::class)->track(
                        'CompletePayment',
                        $tiktokUserData,
                        $tiktokProperties,
                        $order->order_number
                    );
                } catch (\Exception $e) {
                    \Log::error('TikTok CAPI CompletePayment Failed (SSLCommerz): ' . $e->getMessage());
                }

                return redirect()->route('checkout.success', ['order' => $order->order_number])->with('success', 'Payment Successful via SSLCommerz');
            } else {
                return redirect()->route('checkout.success', ['order' => $order->order_number])->with('error', 'Payment Validation Failed: Amount Mismatch');
            }
        } elseif ($status == 'FAILED') {
            $order->payment_status = 'failed';
            $order->save();
            return redirect()->route('checkout.success', ['order' => $order->order_number])->with('error', 'Payment Failed.');
        } elseif ($status == 'CANCELLED') {
            return redirect()->route('checkout.success', ['order' => $order->order_number])->with('error', 'Payment Cancelled.');
        }

        return redirect()->route('checkout.success', ['order' => $order->order_number])->with('error', 'Invalid Payment Status.');
    }
}
