<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Color;
use App\Models\Size;
use App\Services\FraudCheckService;
use App\Services\SmsService;
use App\Services\FacebookConversionService;
use App\Services\TiktokConversionService;
use App\Models\ShippingCharge;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class CheckoutController extends Controller
{
    public function index(Request $request)
    {
        $cart = [];

        // Handle Direct Checkout (Buy Now) from URL parameters or Old Input (Redirect Back)
        $productId = $request->input('product_id') ?? $request->old('product_id');
        
        if ($productId) {
            $product = Product::find($productId);
            if ($product) {
                $price = $product->offer_price ?? $product->price;
                $variant = null;

                // Check for variation
                $colorName = null;
                $sizeName = null;

                $colorId = $request->input('color_id') ?? $request->old('color_id');
                $sizeId = $request->input('size_id') ?? $request->old('size_id');
                $quantity = $request->input('quantity') ?? $request->old('quantity', 1);

                if ($colorId) {
                    $color = Color::find($colorId);
                    if ($color) $colorName = $color->name;
                }
                if ($sizeId) {
                    $size = Size::find($sizeId);
                    if ($size) $sizeName = $size->name;
                }

                if ($colorId || $sizeId) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($colorId) $query->where('color_id', $colorId);
                    if ($sizeId) $query->where('size_id', $sizeId);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $price,
                    'quantity' => $quantity,
                    'thumbnail' => $variant && $variant->image ? $variant->image : $product->thumbnail,
                    'color_id' => $colorId,
                    'size_id' => $sizeId,
                    'color_name' => $colorName,
                    'size_name' => $sizeName,
                ];

                // Fire AddToCart Event for Direct Checkout (Buy Now)
                try {
                    $addToCartEventId = (string) Str::uuid();
                    $user = Auth::user();
                    $userData = [
                        'ip' => request()->ip(),
                        'user_agent' => request()->userAgent(),
                    ];
                    if ($user) {
                        $userData['email'] = $user->email;
                        $userData['phone'] = $user->phone;
                        $userData['external_id'] = (string)$user->id;
                    }

                    // TikTok AddToCart
                    $tiktokAddToCartProps = [
                        'currency' => 'BDT',
                        'value' => $price * $quantity,
                        'content_type' => 'product',
                        'content_id' => (string)$product->id,
                        'content_name' => $product->name,
                        'quantity' => $quantity,
                    ];
                    
                    app(TiktokConversionService::class)->track(
                        'AddToCart',
                        $userData,
                        $tiktokAddToCartProps,
                        $addToCartEventId
                    );
                    
                    session()->flash('tiktok_event', [
                        'name' => 'AddToCart',
                        'params' => array_merge($tiktokAddToCartProps, ['event_id' => $addToCartEventId])
                    ]);

                    // Facebook AddToCart
                    $fbAddToCartProps = [
                        'currency' => 'BDT',
                        'value' => $price * $quantity,
                        'content_type' => 'product',
                        'content_ids' => [(string)$product->id],
                        'content_name' => $product->name,
                        'content_category' => $product->category ? $product->category->name : null,
                        'num_items' => $quantity
                    ];

                    app(FacebookConversionService::class)->track(
                        'AddToCart',
                        $userData,
                        $fbAddToCartProps,
                        $addToCartEventId
                    );

                    session()->flash('facebook_event', [
                        'name' => 'AddToCart',
                        'params' => array_merge($fbAddToCartProps, ['event_id' => $addToCartEventId])
                    ]);

                } catch (\Exception $e) {
                    \Log::error('Direct Checkout AddToCart Event Failed: ' . $e->getMessage());
                }
            }
        } else {
            // Normal Cart Checkout
            $cart = Session::get('cart', []);
        }

        if (count($cart) == 0) {
            return redirect()->route('shop')->with('error', 'Your cart is empty');
        }

        $total = 0;
        $tiktokContents = [];
        foreach ($cart as $key => $item) {
            $total += $item['price'] * $item['quantity'];
            
            $productId = $item['product_id'] ?? $key;
            if (!isset($item['product_id']) && strpos($key, '_') !== false) {
               $productId = explode('_', $key)[0];
            }
            $tiktokContents[] = [
               'content_id' => (string)$productId,
               'quantity' => $item['quantity'],
               'price' => $item['price'],
               'content_name' => $item['name'] ?? 'Product',
            ];
        }

        $settings = \App\Models\GeneralSetting::first();
        $shippingCharges = ShippingCharge::where('status', true)->get();
        
        $eventId = (string) Str::uuid();

        // Fire TikTok & Facebook CAPI InitiateCheckout Event
        try {
            $user = Auth::user();
            $userData = [
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ];
            
            if ($user) {
                $userData['email'] = $user->email;
                $userData['phone'] = $user->phone;
                $userData['external_id'] = (string)$user->id;
            }

            $tiktokProperties = [
                'currency' => 'BDT',
                'value' => $total,
                'content_type' => 'product',
                'contents' => $tiktokContents,
            ];

            // TikTok
            app(TiktokConversionService::class)->track(
                'InitiateCheckout',
                $userData,
                $tiktokProperties,
                $eventId
            );

            // Facebook
            $facebookContents = [];
            $contentIds = [];
            foreach ($tiktokContents as $item) {
                $facebookContents[] = [
                    'id' => $item['content_id'],
                    'quantity' => $item['quantity'],
                    'item_price' => $item['price'],
                    'title' => $item['content_name']
                ];
                $contentIds[] = $item['content_id'];
            }

            $facebookCustomData = [
                'currency' => 'BDT',
                'value' => $total,
                'content_type' => 'product',
                'contents' => $facebookContents,
                'content_ids' => $contentIds,
                'num_items' => count($facebookContents)
            ];

            app(FacebookConversionService::class)->track(
                'InitiateCheckout',
                $userData,
                $facebookCustomData,
                $eventId
            );
        } catch (\Exception $e) {
            \Log::error('CAPI InitiateCheckout Event Failed: ' . $e->getMessage());
        }

        return view('frontend.checkout.index', compact('cart', 'total', 'settings', 'eventId', 'shippingCharges'));
    }

    public function calculateShipping(Request $request)
    {
        $request->validate([
            'area_id' => 'required|exists:shipping_charges,id',
        ]);

        $areaId = $request->input('area_id');
        
        // Prepare Cart
        $cart = [];
        if ($request->filled('product_id')) {
             $product = Product::find($request->product_id);
             if ($product) {
                $price = $product->offer_price ?? $product->price;
                $variant = null;

                // Check for variation logic for shipping calculation
                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'quantity' => $request->quantity ?? 1,
                    // We need correct price for total calculation in response
                    'price' => $price,
                    'color_id' => $request->color_id,
                    'size_id' => $request->size_id,
                ];
             }
        } else {
             $cart = Session::get('cart', []);
        }

        $cost = $this->_calculateShippingCost($areaId, $cart);
        
        $subtotal = 0;
        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? explode('_', $key)[0];
            $product = Product::find($productId);
            
            $quantity = $item['quantity'];
            $price = isset($item['price']) ? $item['price'] : ($product->price ?? 0);
            if (isset($item['price'])) $subtotal += $price * $quantity;
            else $subtotal += ($product ? $product->offer_price ?? $product->price : 0) * $quantity;
        }
        
        return response()->json([
            'cost' => $cost, 
            'total' => $subtotal + $cost,
            'formatted_cost' => '৳' . number_format($cost, 0),
            'formatted_total' => '৳' . number_format($subtotal + $cost, 0)
        ]);
    }

    private function _calculateShippingCost($areaId, $cart)
    {
        $shippingCharge = ShippingCharge::find($areaId);
        if (!$shippingCharge) {
            return 0;
        }

        $totalWeight = 0;
        $extraShippingCost = 0;

        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? explode('_', $key)[0];
            $product = Product::find($productId);
            
            $quantity = $item['quantity'];

                if ($product) {
                $weight = $product->weight;
                $weightUnit = $product->weight_unit;
                $variantShippingCost = null;
                $deliveryType = 'flat_rate'; // Default

                // Check for variant
                if (isset($item['color_id']) || isset($item['size_id'])) {
                    $variantQuery = ProductVariant::where('product_id', $productId);
                    if (isset($item['color_id'])) {
                        $variantQuery->where('color_id', $item['color_id']);
                    }
                    if (isset($item['size_id'])) {
                        $variantQuery->where('size_id', $item['size_id']);
                    }
                    $variant = $variantQuery->first();

                    if ($variant) {
                        if ($variant->weight) {
                            $weight = $variant->weight;
                            $weightUnit = $variant->weight_unit;
                        }
                        if ($variant->shipping_cost) {
                            $variantShippingCost = $variant->shipping_cost;
                        }
                        if ($variant->delivery_type) {
                            $deliveryType = $variant->delivery_type;
                        }
                    }
                }

                if ($deliveryType == 'free' || $deliveryType == 'pickup') {
                    // No shipping cost for this item
                    continue; 
                }

                if ($variantShippingCost !== null && $variantShippingCost > 0) {
                     $extraShippingCost += ($variantShippingCost * $quantity);
                } else {
                    if ($weight) {
                        // Normalize to KG
                        if (in_array(strtolower($weightUnit), ['gm', 'g', 'gram', 'grams'])) {
                             $weight = $weight / 1000;
                        }
                        $totalWeight += ($weight * $quantity);
                    }
                }
            }
        }
        
        // Default weight if 0 (e.g. 0.5kg base) ONLY if there are items contributing to weight
        // If all items have custom shipping cost, totalWeight might be 0.
        
        $weightCost = 0;
        if ($totalWeight > 0) {
             if ($totalWeight <= 0.5) {
                $weightCost = $shippingCharge->charge_0_0_5;
            } elseif ($totalWeight <= 1) {
                $weightCost = $shippingCharge->charge_0_5_1;
            } elseif ($totalWeight <= 2) {
                $weightCost = $shippingCharge->charge_1_2;
            } else {
                // Over 2kg logic: Base charge (Charge for 2kg range) + Extra per kg
                $weightCost = $shippingCharge->charge_1_2 + (ceil($totalWeight - 2) * $shippingCharge->charge_over_2);
            }
        } elseif ($extraShippingCost == 0 && count($cart) > 0) {
            // Fallback if weight is 0 and no extra cost (maybe product has no weight set)
             $weightCost = $shippingCharge->charge_0_0_5;
        }
        
        return $weightCost + $extraShippingCost;
    }

    public function storeIncomplete(Request $request)
    {
        // Custom validation for Bangladeshi phone number
        if (!preg_match('/^01[3-9]\d{8}$/', $request->phone)) {
            return response()->json(['status' => 'ignored']);
        }

        // Check recent real order to prevent spam
        // TEMPORARILY DISABLED
        /*
        $recentRealOrder = Order::where('phone', $request->phone)
            ->where('status', '!=', 'incomplete')
            ->where('status', '!=', 'cancelled')
            ->where('created_at', '>=', now()->subMinutes(15))
            ->first();

        if ($recentRealOrder) {
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            return response()->json(['status' => 'ignored_recent_purchase']);
        }
        */

        $request->validate([
            'phone' => 'required|string|max:20',
        ]);

        $cart = [];
        if ($request->filled('product_id')) {
             $product = Product::find($request->product_id);
             if ($product) {
                $price = $product->offer_price ?? $product->price;
                
                // Variation Check for Incomplete Order
                if ($request->filled('color_id') || $request->filled('size_id')) {
                    $query = ProductVariant::where('product_id', $product->id);
                    if ($request->filled('color_id')) $query->where('color_id', $request->color_id);
                    if ($request->filled('size_id')) $query->where('size_id', $request->size_id);
                    $variant = $query->first();

                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'quantity' => $request->quantity ?? 1,
                    'price' => $price,
                    'name' => $product->name,
                    'color_id' => $request->filled('color_id') ? $request->color_id : null,
                    'size_id' => $request->filled('size_id') ? $request->size_id : null,
                ];
             }
        } else {
             $cart = Session::get('cart', []);
        }

        if (count($cart) == 0) {
            return response()->json(['status' => 'empty_cart']);
        }

        // Calculate Shipping
        $shippingCost = 0;
        if ($request->filled('area_id')) {
            $shippingCost = $this->_calculateShippingCost($request->area_id, $cart);
        }

        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        $total += $shippingCost;

        $orderId = $request->input('incomplete_order_id') ?? Session::get('incomplete_order_id');
        $order = null;

        if ($orderId) {
            $order = Order::where('id', $orderId)->where('status', 'incomplete')->first();
        }

        if (!$order) {
            $existingOrders = Order::where('phone', $request->phone)
                          ->where('status', 'incomplete')
                          ->orderBy('id', 'desc')
                          ->get();

            if ($existingOrders->count() > 0) {
                $order = $existingOrders->first();
                if ($existingOrders->count() > 1) {
                    Order::whereIn('id', $existingOrders->slice(1)->pluck('id'))->delete();
                }
            } else {
                $order = new Order();
                $order->order_number = 'INC-' . strtoupper(Str::random(10));
                $order->status = 'incomplete';
            }
        }

        $order->user_id = Auth::id();
        $order->first_name = $request->first_name ?? 'Guest';
        $order->last_name = $request->last_name ?? '';
        $order->email = $request->email ?? '';
        $order->phone = $request->phone;
        $order->address = $request->address ?? '';
        $order->city = $request->city ?? '';
        $order->postal_code = $request->postal_code ?? '';
        $order->delivery_cost = $shippingCost;
        $order->total_amount = $total;
        $order->payment_method = $request->payment_method ?? 'cod';
        $order->payment_status = 'pending';
        $order->source = $request->source ?? 'website';
        $order->save();

        Session::put('incomplete_order_id', $order->id);

        Order::where('phone', $request->phone)
             ->where('status', 'incomplete')
             ->where('id', '!=', $order->id)
             ->delete();

        try {
            app(FraudCheckService::class)->checkAndSave($request->phone);
        } catch (\Throwable $e) {
            // Ignore
        }

        $order->items()->delete();

        foreach ($cart as $key => $item) {
            $productId = $item['product_id'] ?? $key;
            if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                $productId = explode('_', $key)[0];
            }

            OrderItem::create([
                'order_id' => $order->id,
                'product_id' => $productId,
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'color_id' => $item['color_id'] ?? null,
                'size_id' => $item['size_id'] ?? null,
            ]);
        }

        return response()->json(['status' => 'success', 'order_id' => $order->id]);
    }

    public function store(Request $request)
    {
        \Log::info('Checkout store method called', $request->all());

        // STRICTLY REQUIRED FIELDS: Phone, Address, Area ID
        $rules = [
            'phone' => 'required|string|max:20',
            'address' => 'required|string|max:255',
            'area_id' => 'required|exists:shipping_charges,id',
        ];

        // OPTIONAL FIELDS: Everything else (Name, Email, City, etc.)
        // We set them to nullable to ensure no validation error blocks submission
        $rules['name'] = 'nullable|string|max:255';
        $rules['first_name'] = 'nullable|string|max:255';
        $rules['last_name'] = 'nullable|string|max:255';
        $rules['email'] = 'nullable|email|max:255';
        $rules['city'] = 'nullable|string|max:255';
        $rules['postal_code'] = 'nullable|string|max:20';
        // Variation Validation
        $rules['color_id'] = 'nullable|integer';
        $rules['size_id'] = 'nullable|integer';

        $request->validate($rules);

        try {
            DB::beginTransaction();
            \Log::info('Checkout transaction started');

            $cart = [];
            
            // Handle Direct Checkout (e.g. from Landing Page)
            if ($request->has('product_id')) {
                $product = Product::find($request->product_id);
                if (!$product) {
                    return back()->with('error', 'Product not found');
                }
                $price = $product->offer_price ?? $product->price;
                
                // Sanitize Variation Inputs
                $colorId = $request->input('color_id');
                if ($colorId === '' || $colorId === 'null') $colorId = null;
                
                $sizeId = $request->input('size_id');
                if ($sizeId === '' || $sizeId === 'null') $sizeId = null;

                // If variation selected, get correct price
                if ($colorId || $sizeId) {
                    $variant = ProductVariant::where('product_id', $product->id)
                        ->when($colorId, function($q) use ($colorId) { $q->where('color_id', $colorId); })
                        ->when($sizeId, function($q) use ($sizeId) { $q->where('size_id', $sizeId); })
                        ->first();
                    
                    if ($variant) {
                        $price = $variant->offer_price ?? $variant->price;
                    }
                }

                $cart[] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $price,
                    'quantity' => $request->quantity ?? 1,
                    'color_id' => $colorId,
                    'size_id' => $sizeId,
                ];
            } else {
                // Standard Cart Checkout
                $cart = Session::get('cart', []);
                if (count($cart) == 0) {
                    \Log::warning('Cart is empty during checkout');
                    return redirect()->route('shop')->with('error', 'Your cart is empty');
                }
            }

            // Validate Stock
            foreach ($cart as $key => $item) {
                $productId = $item['product_id'] ?? explode('_', $key)[0];
                $product = Product::find($productId);
                
                if (!$product) {
                    \Log::error('Product not found: ' . $item['name']);
                    return redirect()->back()->with('error', 'Product not found: ' . $item['name'])->withInput();
                }

                $quantity = $item['quantity'];
                $colorId = $item['color_id'] ?? null;
                $sizeId = $item['size_id'] ?? null;

                // Check Variant Stock
                if ($colorId || $sizeId) {
                    $variant = ProductVariant::where('product_id', $productId)
                        ->when($colorId, function($q) use ($colorId) { $q->where('color_id', $colorId); })
                        ->when($sizeId, function($q) use ($sizeId) { $q->where('size_id', $sizeId); })
                        ->first();
                    
                    if ($variant && $variant->stock_quantity < $quantity) {
                        \Log::error('Insufficient stock for variant', ['product' => $product->name]);
                        return redirect()->back()->with('error', 'Insufficient stock for ' . $product->name . ' (Variant)')->withInput();
                    }
                }

                // Check Main Product Stock
                if ($product->stock_quantity < $quantity) {
                    \Log::error('Insufficient stock for product', ['product' => $product->name]);
                    return redirect()->back()->with('error', 'Insufficient stock for ' . $product->name)->withInput();
                }
            }

            // Calculate Shipping
            $shippingCost = $this->_calculateShippingCost($request->area_id, $cart);

            $total = 0;
            foreach ($cart as $item) {
                $total += $item['price'] * $item['quantity'];
            }
            $total += $shippingCost;

            // Prevent duplicate orders (Same Phone + Same Product + Within 24 Hours)
            // TEMPORARILY DISABLED FOR TESTING/DEVELOPMENT
            /*
            $checkTime = now()->subHours(24);
            $recentOrders = Order::where('phone', $request->phone)
                ->where('status', '!=', 'cancelled')
                ->where('status', '!=', 'incomplete')
                ->where('created_at', '>=', $checkTime)
                ->with('items')
                ->orderBy('created_at', 'desc')
                ->get();

            if ($recentOrders->count() > 0) {
                // Get product IDs from current request
                $currentProductIds = [];
                foreach ($cart as $key => $item) {
                    $pid = $item['product_id'] ?? explode('_', $key)[0];
                    $currentProductIds[] = (string)$pid;
                }

                foreach ($recentOrders as $recentOrder) {
                    $recentProductIds = $recentOrder->items->pluck('product_id')->map(function($id) { return (string)$id; })->toArray();
                    
                    // Check intersection
                    if (count(array_intersect($currentProductIds, $recentProductIds)) > 0) {
                        DB::rollBack();
                        
                        $timeAgo = $recentOrder->created_at->diffForHumans();
                        $contactSetting = \App\Models\ContactSetting::first();
                        $supportPhone = $contactSetting ? ($contactSetting->phone_number ?? 'our support') : 'our support';
                        
                        return redirect()->back()->with('duplicate_error', [
                            'message_en' => "You have already ordered this product {$timeAgo}. If you want to order again, please contact us.",
                            'message_bn' => "আপনি ইতিমধ্যে {$timeAgo} এই পণ্যটি অর্ডার করেছেন। আপনি যদি আবার অর্ডার করতে চান তবে অনুগ্রহ করে আমাদের সাথে যোগাযোগ করুন।",
                            'phone' => $supportPhone
                        ]);
                    }
                }
            }
            */

            // Remove any incomplete orders
            $incompleteId = $request->input('incomplete_order_id') ?? Session::get('incomplete_order_id');
            if ($incompleteId) {
                Order::where('id', $incompleteId)->delete();
                Session::forget('incomplete_order_id');
            }
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            \Log::info('Incomplete orders deleted');

            // Handle Name Splitting if needed
            $firstName = $request->first_name;
            $lastName = $request->last_name;
            if (!$firstName && $request->name) {
                $parts = explode(' ', $request->name, 2);
                $firstName = $parts[0];
                $lastName = $parts[1] ?? '-';
            }

            // Fallback for optional name
            if (empty($firstName)) {
                $firstName = 'Guest';
                $lastName = '-';
            }

            // Create Order
            $orderData = [
                'order_number' => 'ORD-' . strtoupper(Str::random(10)),
                'user_id' => Auth::id(), // Nullable
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $request->email ?? '',
                'phone' => $request->phone,
                'address' => $request->address,
                'city' => $request->city ?? '',
                'postal_code' => $request->postal_code ?? '',
                'area_id' => $request->area_id,
                'delivery_cost' => $shippingCost,
                'total_amount' => $total,
                'payment_method' => $request->payment_method ?? 'cod',
                'payment_status' => 'pending',
                'source' => $request->source ?? 'website',
            ];
            \Log::info('Creating order', $orderData);
            
            $order = Order::create($orderData);
            \Log::info('Order created: ' . $order->id);

            // Send SMS Notification
            try {
                app(SmsService::class)->sendOrderConfirmation($order);
            } catch (\Throwable $e) {
                \Log::error('Failed to send order SMS: ' . $e->getMessage());
            }

            // Perform Fraud Check
            try {
                \Log::info('Starting fraud check');
                $fraudResult = app(FraudCheckService::class)->checkAndSave($request->phone);
                if (!$fraudResult['success']) {
                    \Log::error('Fraud Check Service Error: ' . ($fraudResult['message'] ?? 'Unknown error'));
                } else {
                    $order->fraud_score = $fraudResult['percentage'];
                    $order->fraud_data = $fraudResult['data'];
                    $order->fraud_checked_at = now();
                    $order->save();
                }
            } catch (\Throwable $e) {
                \Illuminate\Support\Facades\Log::error('Auto Fraud Check Failed: ' . $e->getMessage());
            }

            // Create Order Items and Decrement Stock
            foreach ($cart as $key => $item) {
                $productId = $item['product_id'] ?? $key;
                if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                    $productId = explode('_', $key)[0];
                }

                $quantity = $item['quantity'];
                $colorId = $item['color_id'] ?? null;
                $sizeId = $item['size_id'] ?? null;

                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $productId,
                    'quantity' => $quantity,
                    'price' => $item['price'],
                    'color_id' => $colorId,
                    'size_id' => $sizeId,
                ]);

                // Decrement Stock
                $product = Product::find($productId);
                if ($product) {
                    $product->decrement('stock_quantity', $quantity);
                    
                    if ($colorId || $sizeId) {
                        $variant = ProductVariant::where('product_id', $productId)
                            ->when($colorId, function($q) use ($colorId) { $q->where('color_id', $colorId); })
                            ->when($sizeId, function($q) use ($sizeId) { $q->where('size_id', $sizeId); })
                            ->first();
                        
                        if ($variant) {
                            $variant->decrement('stock_quantity', $quantity);
                        }
                    }
                }
            }
            \Log::info('Order items created');

            // Clear Cart
            Session::forget('cart');
            
            // Delete incomplete orders
            $incompleteId = Session::get('incomplete_order_id') ?? $request->input('incomplete_order_id');
            if ($incompleteId) {
                Order::where('id', $incompleteId)->where('status', 'incomplete')->delete();
                Session::forget('incomplete_order_id');
            }
            Order::where('phone', $request->phone)->where('status', 'incomplete')->delete();
            
            Session::save();
            
            DB::commit();
            \Log::info('Transaction committed');

            // Fire Facebook CAPI Purchase Event
            $eventId = (string) Str::uuid();
            try {
                $userData = [
                    'email' => $order->email,
                    'phone' => $order->phone,
                    'first_name' => $order->first_name,
                    'last_name' => $order->last_name,
                    'city' => $order->city,
                    'zip' => $order->postal_code,
                    'country' => 'BD',
                    'external_id' => $order->user_id ? (string)$order->user_id : null,
                    'ip' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ];

                $contents = [];
                $contentIds = [];
                foreach ($cart as $key => $item) {
                    $productId = $item['product_id'] ?? $key;
                    if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                       $productId = explode('_', $key)[0];
                   }
                   $contents[] = [
                       'id' => (string)$productId,
                       'quantity' => $item['quantity'],
                       'item_price' => $item['price'],
                       'title' => $item['name'] ?? 'Product',
                   ];
                   $contentIds[] = (string)$productId;
               }

                $customData = [
                    'currency' => 'BDT',
                    'value' => $order->total_amount,
                    'content_type' => 'product',
                    'contents' => $contents,
                    'content_ids' => $contentIds,
                    'order_id' => $order->order_number,
                ];

                app(FacebookConversionService::class)->track(
                    'Purchase',
                    $userData,
                    $customData,
                    $eventId
                );

                // TikTok CAPI
                $tiktokContents = [];
                foreach ($cart as $key => $item) {
                     $productId = $item['product_id'] ?? $key;
                     if (!isset($item['product_id']) && strpos($key, '_') !== false) {
                        $productId = explode('_', $key)[0];
                    }
                    $tiktokContents[] = [
                       'content_id' => (string)$productId,
                       'quantity' => $item['quantity'],
                       'price' => $item['price'],
                       'content_name' => $item['name'] ?? 'Product',
                    ];
                }

                $tiktokProperties = [
                    'currency' => 'BDT',
                    'value' => $order->total_amount,
                    'content_type' => 'product',
                    'contents' => $tiktokContents,
                ];

                app(TiktokConversionService::class)->track(
                    'PlaceAnOrder',
                    $userData,
                    $tiktokProperties,
                    $eventId
                );

            } catch (\Exception $e) {
                \Log::error('CAPI Purchase Event Failed: ' . $e->getMessage());
            }

            return redirect()->route('checkout.success', $order->id)->with('success', 'Order placed successfully!')->with('purchase_event_id', $eventId);

        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Checkout Failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong. Please try again. ' . $e->getMessage())->withInput();
        }
    }

    public function success($id)
    {
        $order = Order::findOrFail($id);
        $eventId = session('purchase_event_id');
        
        // Optional: Check if the order belongs to the authenticated user if logged in
        if (Auth::check() && $order->user_id && $order->user_id !== Auth::id()) {
            abort(403);
        }

        return view('frontend.checkout.success', compact('order', 'eventId'));
    }
}
