<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use App\Services\FacebookConversionService;
use App\Services\TiktokConversionService;
use Illuminate\Support\Str;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        event(new Registered($user));

        Auth::login($user);

        // Fire CompleteRegistration Event
        try {
            $eventId = (string) Str::uuid();
            
            $userData = [
                'email' => $user->email,
                'external_id' => (string)$user->id,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ];

            // Facebook CAPI
            app(FacebookConversionService::class)->track(
                'CompleteRegistration',
                $userData,
                ['status' => true, 'content_name' => 'Registration'],
                $eventId
            );

            // TikTok CAPI
            app(TiktokConversionService::class)->track(
                'CompleteRegistration',
                $userData,
                ['status' => true],
                $eventId
            );

            // Flash for Browser Pixel
            session()->flash('facebook_event', [
                'name' => 'CompleteRegistration',
                'params' => ['status' => true, 'event_id' => $eventId]
            ]);
            
            session()->flash('tiktok_event', [
                'name' => 'CompleteRegistration',
                'params' => ['status' => true, 'event_id' => $eventId]
            ]);

        } catch (\Exception $e) {
            \Log::error('CAPI CompleteRegistration Event Failed: ' . $e->getMessage());
        }

        return redirect(route('home', absolute: false));
    }
}
