<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\ShippingCharge;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    public function general()
    {
        $setting = GeneralSetting::first() ?? new GeneralSetting();
        return view('admin.settings.general', compact('setting'));
    }

    public function shipping()
    {
        $shippingCharges = ShippingCharge::all();
        $setting = GeneralSetting::first() ?? new GeneralSetting();
        return view('admin.settings.shipping', compact('shippingCharges', 'setting'));
    }

    public function updateShipping(Request $request)
    {
        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }

        $request->validate([
            'free_shipping' => 'nullable|boolean',
            'flat_rate_enabled' => 'nullable|boolean',
            'flat_rate_cost' => 'nullable|numeric',
            'charges' => 'nullable|array',
            'charges.*.area_name' => 'required|string|max:255',
            'charges.*.charge_0_0_5' => 'required|numeric|min:0',
            'charges.*.charge_0_5_1' => 'required|numeric|min:0',
            'charges.*.charge_1_2' => 'required|numeric|min:0',
            'charges.*.charge_over_2' => 'required|numeric|min:0',
        ]);

        $setting->free_shipping = $request->has('free_shipping');
        $setting->flat_rate_enabled = $request->has('flat_rate_enabled');
        $setting->flat_rate_cost = $request->input('flat_rate_cost');
        
        $setting->save();

        // Handle Shipping Charges
        // Since we are submitting the whole form, we can sync the charges.
        // Or we can update existing ones and create new ones.
        // For simplicity, let's look at the logic.
        
        $submittedCharges = $request->input('charges', []);
        
        // If no ID, it's a new charge. If ID, update.
        // But if an ID is missing from submission but exists in DB, should we delete it?
        // The screenshot has a trash button. If the user deletes it in UI, it won't be in the form submission.
        // So we should sync: keep provided IDs, delete missing IDs, create new ones (no ID).
        
        $existingIds = ShippingCharge::pluck('id')->toArray();
        $submittedIds = [];

        foreach ($submittedCharges as $key => $chargeData) {
            if (isset($chargeData['id']) && $chargeData['id']) {
                $submittedIds[] = $chargeData['id'];
                ShippingCharge::where('id', $chargeData['id'])->update([
                    'area_name' => $chargeData['area_name'],
                    'charge_0_0_5' => $chargeData['charge_0_0_5'],
                    'charge_0_5_1' => $chargeData['charge_0_5_1'],
                    'charge_1_2' => $chargeData['charge_1_2'],
                    'charge_over_2' => $chargeData['charge_over_2'],
                ]);
            } else {
                $newCharge = ShippingCharge::create([
                    'area_name' => $chargeData['area_name'],
                    'charge_0_0_5' => $chargeData['charge_0_0_5'],
                    'charge_0_5_1' => $chargeData['charge_0_5_1'],
                    'charge_1_2' => $chargeData['charge_1_2'],
                    'charge_over_2' => $chargeData['charge_over_2'],
                    'status' => true,
                ]);
                $submittedIds[] = $newCharge->id;
            }
        }

        // Delete charges that were not in the submission
        // Only if the request was meant to update charges (which it is, if we use one big form)
        // But we need to be careful. If the user submits an empty charges array, it means delete all?
        // Yes, if they deleted all cards.
        
        ShippingCharge::whereNotIn('id', $submittedIds)->delete();

        return back()->with('success', 'Shipping settings updated successfully.');
    }

    public function pixels()
    {
        return view('admin.settings.pixels');
    }

    public function social()
    {
        return view('admin.settings.social');
    }

    public function contact()
    {
        return view('admin.settings.contact');
    }

    public function orderStatus()
    {
        return view('admin.settings.order_status');
    }
    
    public function update(Request $request)
    {
        // Log the request data for debugging
        \Log::info('Setting Update Request:', $request->all());
        \Log::info('Files:', $request->allFiles());

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'facebook_verification_code' => 'nullable|string',
            'google_verification_code' => 'nullable|string',
            'white_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'dark_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'favicon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'og_banner_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'hot_deal_end_date' => 'nullable|date',
            'flash_sale_end_date' => 'nullable|date',
            'header_code' => 'nullable|string',
            'meta_description' => 'nullable|string',
            'meta_keyword' => 'nullable|string',
            'top_headline' => 'nullable|string',
            'checkout_note' => 'nullable|string',
            'order_policy' => 'nullable|string',
            'all_products_at_home' => 'nullable|boolean',
            'categories_wise_products_at_home' => 'nullable|boolean',
            'status' => 'nullable|boolean',
            'show_developer_credit' => 'nullable|boolean',
        ]);
        
        // Handle file uploads
        if ($request->hasFile('white_logo')) {
            if ($setting->white_logo) Storage::delete($setting->white_logo);
            $data['white_logo'] = $request->file('white_logo')->store('settings');
        }
        
        if ($request->hasFile('dark_logo')) {
            if ($setting->dark_logo) Storage::delete($setting->dark_logo);
            $data['dark_logo'] = $request->file('dark_logo')->store('settings');
        }
        
        if ($request->hasFile('favicon')) {
            if ($setting->favicon) Storage::delete($setting->favicon);
            $data['favicon'] = $request->file('favicon')->store('settings');
        }
        
        if ($request->hasFile('og_banner_image')) {
            if ($setting->og_banner_image) Storage::delete($setting->og_banner_image);
            $data['og_banner_image'] = $request->file('og_banner_image')->store('settings');
        }

        $data['all_products_at_home'] = $request->has('all_products_at_home');
        $data['categories_wise_products_at_home'] = $request->has('categories_wise_products_at_home');
        $data['status'] = $request->has('status');
        $data['show_developer_credit'] = $request->has('show_developer_credit');

        $setting->fill($data);
        $setting->save();

        return back()->with('success', 'Settings updated successfully.');
    }
}
