<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Supplier;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $purchases = Purchase::with('supplier')->latest()->paginate(10);
        return view('admin.purchases.index', compact('purchases'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $suppliers = Supplier::where('status', true)->get();
        $products = Product::select('id', 'name', 'thumbnail', 'stock_quantity', 'price')->get();
        return view('admin.purchases.create', compact('suppliers', 'products'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'purchase_date' => 'required|date',
            'products' => 'required|array|min:1',
            'products.*.id' => 'required|exists:products,id',
            'products.*.quantity' => 'required|integer|min:1',
            'products.*.unit_cost' => 'required|numeric|min:0',
        ]);

        DB::transaction(function () use ($request) {
            $data = $request->only(['supplier_id', 'purchase_date', 'note']);
            $data['reference_no'] = 'PUR-' . strtoupper(Str::random(8));
            $data['status'] = 'pending';
            
            // Calculate total amount
            $totalAmount = 0;
            foreach ($request->products as $item) {
                $totalAmount += $item['quantity'] * $item['unit_cost'];
            }
            $data['total_amount'] = $totalAmount;

            $purchase = Purchase::create($data);

            foreach ($request->products as $item) {
                $product = Product::find($item['id']);
                $purchase->items()->create([
                    'product_id' => $item['id'],
                    'product_name' => $product->name,
                    'quantity' => $item['quantity'],
                    'unit_cost' => $item['unit_cost'],
                    'subtotal' => $item['quantity'] * $item['unit_cost'],
                ]);
            }
        });

        return redirect()->route('admin.purchases.index')->with('success', 'Purchase created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $purchase = Purchase::with(['supplier', 'items.product'])->findOrFail($id);
        return view('admin.purchases.show', compact('purchase'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $purchase = Purchase::with('items')->findOrFail($id);
        $suppliers = Supplier::where('status', true)->get();
        $products = Product::select('id', 'name', 'thumbnail', 'stock_quantity', 'price')->get();
        return view('admin.purchases.edit', compact('purchase', 'suppliers', 'products'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // For simplicity, not implementing full update logic with item syncing right now unless requested.
        // Usually involves deleting old items and re-creating, or diffing.
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'purchase_date' => 'required|date',
            'status' => 'required|in:pending,completed,cancelled',
        ]);

        $purchase = Purchase::findOrFail($id);
        $purchase->update($request->only(['supplier_id', 'purchase_date', 'status', 'note']));

        return redirect()->route('admin.purchases.index')->with('success', 'Purchase updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $purchase = Purchase::findOrFail($id);
        $purchase->delete();

        return redirect()->route('admin.purchases.index')->with('success', 'Purchase deleted successfully.');
    }
}
