<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Color;
use App\Models\Size;
use App\Models\ProductImage;
use App\Models\ProductVariant;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:products.view')->only(['index', 'show']);
        $this->middleware('permission:products.create')->only(['create', 'store']);
        $this->middleware('permission:products.edit')->only(['edit', 'update']);
        $this->middleware('permission:products.delete')->only(['destroy']);
    }

    public function priceEdit()
    {
        return view('admin.products.price_edit');
    }

    public function priceUpdate(Request $request)
    {
        // Logic for bulk price update
        return back()->with('success', 'Prices updated successfully');
    }

    public function search(Request $request)
    {
        $search = $request->input('q');
        $query = Product::query();

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%");
            });
        }

        $products = $query->select('id', 'name', 'sku', 'price', 'thumbnail', 'stock_quantity')
                          ->latest()
                          ->take(20)
                          ->get();
        
        $products->transform(function ($product) {
            $product->thumbnail_url = $product->thumbnail ? Storage::url($product->thumbnail) : null;
            return $product;
        });

        return response()->json($products);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;
        
        $query = Product::with(['category', 'brand']);

        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        }

        $products = $query->latest()->paginate(10);
        return view('admin.products.index', compact('products', 'startDate', 'endDate'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::where('status', true)->get();
        $brands = Brand::where('status', true)->get();
        $colors = Color::where('status', true)->get();
        $sizes = Size::where('status', true)->get();

        return view('admin.products.create', compact('categories', 'brands', 'colors', 'sizes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'price' => 'required|numeric|min:0',
            'purchase_price' => 'nullable|numeric|min:0',
            'weight' => 'required|numeric|min:0',
            'delivery_type' => 'required|string',
            'thumbnail' => 'nullable|image|max:2048',
            'images.*' => 'nullable|image|max:2048',
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['images', 'thumbnail']); // Exclude file inputs to handle manually
            
            // Handle Slug
            if ($request->filled('slug')) {
                $data['slug'] = Str::slug($request->slug);
            } else {
                $data['slug'] = Str::slug($request->name);
            }

            // Handle Boolean Fields (Checkboxes)
            $data['status'] = $request->has('status') ? 1 : 0;
            $data['pre_order'] = $request->has('pre_order') ? 1 : 0;
            $data['is_digital'] = $request->has('is_digital') ? 1 : 0;
            $data['has_seo'] = $request->has('has_seo') ? 1 : 0;
            $data['offer_1'] = $request->has('offer_1') ? 1 : 0;
            $data['offer_2'] = $request->has('offer_2') ? 1 : 0;
            $data['offer_3'] = $request->has('offer_3') ? 1 : 0;
            $data['is_featured'] = $request->has('is_featured') ? 1 : 0; // If you add this field later

            // Handle Thumbnail
            if ($request->hasFile('thumbnail')) {
                $data['thumbnail'] = $request->file('thumbnail')->store('products/thumbnails', 'public');
            }

            $product = Product::create($data);

            // Handle Multiple Images
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products/images', 'public');
                    ProductImage::create([
                        'product_id' => $product->id,
                        'image_path' => $path,
                    ]);
                }
            }

            // Auto-assign thumbnail from first image if missing
            if (empty($product->thumbnail)) {
                $firstImage = $product->images()->first();
                if ($firstImage) {
                    $product->update(['thumbnail' => $firstImage->image_path]);
                }
            }

            // Handle Variants
            if ($request->has('has_variant')) {
                // Ensure Sizes exist with correct codes
                if ($request->has('variant_options_list')) {
                    $sizes = $request->input('variant_options_list');
                    foreach ($sizes as $sizeName) {
                        if (!empty($sizeName)) {
                            $code = Str::slug($sizeName);
                            if (empty($code)) $code = $sizeName;
                            Size::updateOrCreate(['name' => $sizeName], ['code' => $code, 'status' => 1]);
                        }
                    }
                }

                // Ensure Colors exist with correct codes
                if ($request->has('variant_options_color_name')) {
                     $names = $request->input('variant_options_color_name');
                     $codes = $request->input('variant_options_color_code');
                     foreach ($names as $index => $colorName) {
                         if (!empty($colorName)) {
                             $code = $codes[$index] ?? '#000000';
                             Color::updateOrCreate(['name' => $colorName], ['code' => $code, 'status' => 1]);
                         }
                     }
                }

                if ($request->has('variations') && is_array($request->input('variations')) && count($request->input('variations')) > 0) {
                    foreach ($request->input('variations') as $index => $variantData) {
                        $sizeId = null;
                        $colorId = null;

                        if (!empty($variantData['size'])) {
                            $sizeName = $variantData['size'];
                            $sizeCode = Str::slug($sizeName);
                            if (empty($sizeCode)) $sizeCode = $sizeName;
                            
                            $size = Size::firstOrCreate(
                                ['name' => $sizeName], 
                                ['code' => $sizeCode, 'status' => 1]
                            );
                            $sizeId = $size->id;
                        }

                        if (!empty($variantData['color'])) {
                            $colorName = $variantData['color'];
                            $colorCode = '#000000';
                            
                            if (isset($names) && isset($codes)) {
                                $key = array_search($colorName, $names);
                                if ($key !== false) {
                                    $colorCode = $codes[$key] ?? '#000000';
                                }
                            }

                            $color = Color::firstOrCreate(
                                ['name' => $colorName], 
                                ['code' => $colorCode, 'status' => 1]
                            );
                            $colorId = $color->id;
                        }

                        if ($sizeId || $colorId) {
                            $variantPayload = [
                                'product_id' => $product->id,
                                'size_id' => $sizeId,
                                'color_id' => $colorId,
                                'price' => $variantData['price'] ?? 0,
                                'offer_price' => $variantData['offer_price'] ?? 0,
                                'stock_quantity' => $variantData['stock'] ?? 0,
                                'discount_type' => $variantData['discount_type'] ?? null,
                                'discount_value' => $variantData['discount_value'] ?? 0,
                                'inventory_status' => $variantData['inventory_status'] ?? 'in_stock',
                                'weight' => $variantData['weight'] ?? 0,
                                'weight_unit' => $variantData['weight_unit'] ?? 'kg',
                                'description' => $variantData['description'] ?? null,
                                'visibility' => isset($variantData['visibility']) ? 1 : 0,
                                'delivery_type' => $variantData['delivery_type'] ?? 'flat_rate',
                            ];

                            if ($request->hasFile("variations.{$index}.image")) {
                                $path = $request->file("variations.{$index}.image")->store('products/variants', 'public');
                                $variantPayload['image'] = $path;
                            }

                            ProductVariant::create($variantPayload);
                        }
                    }
                }
            }

            DB::commit();

            return redirect()->route('admin.products.index')->with('success', 'Product created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error creating product: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        return view('admin.products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        // Eager load relationships to ensure data is available for the view
        $product->load(['variants.size', 'variants.color', 'colors', 'sizes']);

        $categories = Category::where('status', true)->get();
        $brands = Brand::where('status', true)->get();
        $colors = Color::where('status', true)->get();
        $sizes = Size::where('status', true)->get();
        
        return view('admin.products.edit', compact('product', 'categories', 'brands', 'colors', 'sizes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'price' => 'required|numeric|min:0',
            'purchase_price' => 'nullable|numeric|min:0',
            'weight' => 'required|numeric|min:0',
            'delivery_type' => 'required|string',
            'thumbnail' => 'nullable|image|max:2048',
            'images.*' => 'nullable|image|max:2048',
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['images', 'thumbnail', 'variant_options_list', 'variant_options_color_name', 'variant_options_color_code', 'variant_title_list', 'variant_title_color', 'has_variant', 'variations']);
            
            // Handle Slug
            if ($request->filled('slug')) {
                $data['slug'] = Str::slug($request->slug);
            } else {
                $data['slug'] = Str::slug($request->name);
            }

            // Handle Boolean Fields
            $booleans = ['status', 'pre_order', 'is_digital', 'has_seo', 'offer_1', 'offer_2', 'offer_3', 'is_featured'];
            foreach ($booleans as $field) {
                $data[$field] = $request->has($field) ? 1 : 0;
            }

            // Handle Thumbnail
            if ($request->hasFile('thumbnail')) {
                // Delete old thumbnail
                if ($product->thumbnail) {
                    Storage::disk('public')->delete($product->thumbnail);
                }
                $data['thumbnail'] = $request->file('thumbnail')->store('products/thumbnails', 'public');
            }

            $product->update($data);

            // Handle Multiple Images
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products/images', 'public');
                    ProductImage::create([
                        'product_id' => $product->id,
                        'image_path' => $path,
                    ]);
                }
            }

            // Ensure product has a thumbnail
            if (empty($product->thumbnail)) {
                $firstImage = $product->images()->first();
                if ($firstImage) {
                    $product->update(['thumbnail' => $firstImage->image_path]);
                }
            }

            // Handle Variants
            if ($request->has('has_variant')) {
                // Ensure we have variations data before deleting existing ones
                if ($request->has('variations') && is_array($request->input('variations')) && count($request->input('variations')) > 0) {
                    
                    // Clear existing variants
                    $product->variants()->delete();

                    // Ensure Sizes exist with correct codes
                    if ($request->has('variant_options_list')) {
                        $sizes = $request->input('variant_options_list');
                        foreach ($sizes as $sizeName) {
                            if (!empty($sizeName)) {
                                $code = Str::slug($sizeName);
                                if (empty($code)) $code = $sizeName;
                                Size::updateOrCreate(['name' => $sizeName], ['code' => $code, 'status' => 1]);
                            }
                        }
                    }

                    // Ensure Colors exist with correct codes (Optional, but good for syncing)
                    if ($request->has('variant_options_color_name')) {
                         $names = $request->input('variant_options_color_name');
                         $codes = $request->input('variant_options_color_code');
                         foreach ($names as $index => $colorName) {
                             if (!empty($colorName)) {
                                 $code = $codes[$index] ?? '#000000';
                                 Color::updateOrCreate(['name' => $colorName], ['code' => $code, 'status' => 1]);
                             }
                         }
                    }

                    // Process the variations table
                    foreach ($request->input('variations') as $index => $variantData) {
                        $sizeId = null;
                        $colorId = null;

                        if (!empty($variantData['size'])) {
                            $sizeName = $variantData['size'];
                            $sizeCode = Str::slug($sizeName);
                            if (empty($sizeCode)) $sizeCode = $sizeName;
                            
                            $size = Size::firstOrCreate(
                                ['name' => $sizeName], 
                                ['code' => $sizeCode, 'status' => 1]
                            );
                            $sizeId = $size->id;
                        }

                        if (!empty($variantData['color'])) {
                            // Use firstOrCreate to ensure color exists, preventing null ID
                            $colorName = $variantData['color'];
                            $colorCode = '#000000'; // Default if creating new color without code
                            
                            // Try to find code from request if possible, or fallback
                            if (isset($names) && isset($codes)) {
                                $key = array_search($colorName, $names);
                                if ($key !== false) {
                                    $colorCode = $codes[$key] ?? '#000000';
                                }
                            }

                            $color = Color::firstOrCreate(
                                ['name' => $colorName], 
                                ['code' => $colorCode, 'status' => 1]
                            );
                            $colorId = $color->id;
                        }

                        if ($sizeId || $colorId) {
                            $variantPayload = [
                                'product_id' => $product->id,
                                'size_id' => $sizeId,
                                'color_id' => $colorId,
                                'price' => $variantData['price'] ?? 0,
                                'offer_price' => $variantData['offer_price'] ?? 0,
                                'stock_quantity' => $variantData['stock'] ?? 0,
                                'discount_type' => $variantData['discount_type'] ?? null,
                                'discount_value' => $variantData['discount_value'] ?? 0,
                                'inventory_status' => $variantData['inventory_status'] ?? 'in_stock',
                                'weight' => $variantData['weight'] ?? 0,
                                'weight_unit' => $variantData['weight_unit'] ?? 'kg',
                                'description' => $variantData['description'] ?? null,
                                'visibility' => isset($variantData['visibility']) ? 1 : 0,
                                'delivery_type' => $variantData['delivery_type'] ?? 'flat_rate',
                            ];

                            // Handle Image Upload
                            // Check if a new file is uploaded for this index
                            if ($request->hasFile("variations.{$index}.image")) {
                                $path = $request->file("variations.{$index}.image")->store('products/variants', 'public');
                                $variantPayload['image'] = $path;
                            } elseif (!empty($variantData['existing_image'])) {
                                // Keep existing image if no new one uploaded
                                $variantPayload['image'] = $variantData['existing_image'];
                            }

                            ProductVariant::create($variantPayload);
                        }
                    }
                }
            } elseif ($request->has('has_variant_unchecked')) {
                // Optional: If we had a hidden field to explicitly say "I unchecked it", we would delete.
                // For now, if 'has_variant' is missing, it might mean the checkbox was unchecked.
                // BUT it could also mean the field was missing from request.
                // To be safe, we rely on the checkbox being present to trigger updates.
                // If the user unchecks it, the 'has_variant' is NOT sent.
                // So the block is skipped. Variants REMAIN.
                // This means unchecking the box won't delete variants.
                // To fix this, we need a way to know the user INTENDED to remove variants.
                // Usually a hidden input <input type="hidden" name="has_variant" value="0"> before the checkbox works,
                // but only if the checkbox overrides it.
                // Given the current setup, it's safer to NOT delete if missing, to prevent the bug.
                // Users can manually delete rows in the UI if they want to remove them.
            }

            DB::commit();

            return redirect()->route('admin.products.index')->with('success', 'Product updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error updating product: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        try {
            $product->delete();
            return redirect()->route('admin.products.index')->with('success', 'Product deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Error deleting product: ' . $e->getMessage());
        }
    }

    public function duplicate($id, Request $request)
    {
        $count = (int) ($request->input('count', 1));
        if ($count < 1) {
            $count = 1;
        }

        try {
            DB::beginTransaction();

            $original = Product::with(['images', 'variants'])->findOrFail($id);

            for ($i = 1; $i <= $count; $i++) {
                $copy = $original->replicate();

                $copy->slug = $this->generateUniqueSlug($original->slug ?: $original->name);
                $copy->sku = $this->generateUniqueSku($original->sku);

                $copy->save();

                foreach ($original->images as $img) {
                    ProductImage::create([
                        'product_id' => $copy->id,
                        'image_path' => $img->image_path,
                    ]);
                }

                foreach ($original->variants as $variant) {
                    ProductVariant::create([
                        'product_id' => $copy->id,
                        'size_id' => $variant->size_id,
                        'color_id' => $variant->color_id,
                        'price' => $variant->price,
                        'offer_price' => $variant->offer_price,
                        'stock_quantity' => $variant->stock_quantity,
                        'discount_type' => $variant->discount_type,
                        'discount_value' => $variant->discount_value,
                        'inventory_status' => $variant->inventory_status,
                        'weight' => $variant->weight,
                        'weight_unit' => $variant->weight_unit,
                        'description' => $variant->description,
                        'visibility' => $variant->visibility,
                        'delivery_type' => $variant->delivery_type,
                        'image' => $variant->image,
                    ]);
                }
            }

            DB::commit();

            return back()->with('success', "Product duplicated {$count} time(s).");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error duplicating product: ' . $e->getMessage());
        }
    }

    protected function generateUniqueSlug($base)
    {
        $baseSlug = Str::slug($base);
        $slug = $baseSlug;
        $suffix = 1;
        while (Product::where('slug', $slug)->exists()) {
            $slug = "{$baseSlug}-{$suffix}";
            $suffix++;
        }
        return $slug;
    }

    protected function generateUniqueSku($baseSku)
    {
        if (empty($baseSku)) {
            $sku = strtoupper(Str::random(8));
            while (Product::where('sku', $sku)->exists()) {
                $sku = strtoupper(Str::random(8));
            }
            return $sku;
        }

        $sku = $baseSku . '-' . strtoupper(Str::random(4));
        while (Product::where('sku', $sku)->exists()) {
            $sku = $baseSku . '-' . strtoupper(Str::random(4));
        }
        return $sku;
    }
}
