<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\FacebookPixel;
use App\Models\GtmContainer;
use App\Models\TiktokPixel;
use App\Models\CapiLog;

class PixelController extends Controller
{
    public function index()
    {
        $setting = GeneralSetting::first();
        $pixels = FacebookPixel::all();
        return view('admin.pixels.index', compact('setting', 'pixels'));
    }

    public function logs()
    {
        $logs = CapiLog::orderBy('created_at', 'desc')->paginate(50);
        return view('admin.pixels.logs', compact('logs'));
    }

    public function clearLogs()
    {
        CapiLog::truncate();
        return back()->with('success', 'All CAPI logs cleared successfully.');
    }

    public function tiktok()
    {
        $tiktokPixels = TiktokPixel::all();
        return view('admin.pixels.tiktok', compact('tiktokPixels'));
    }

    public function tagManager()
    {
        $gtmContainers = GtmContainer::all();
        return view('admin.pixels.tag_manager', compact('gtmContainers'));
    }

    public function verification()
    {
        $setting = GeneralSetting::first();
        return view('admin.pixels.verification', compact('setting'));
    }

    public function storeVerification(Request $request)
    {
        $request->validate([
            'facebook_verification_code' => 'nullable|string',
            'tiktok_verification_code' => 'nullable|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }

        $setting->facebook_verification_code = $request->facebook_verification_code;
        $setting->tiktok_verification_code = $request->tiktok_verification_code;
        $setting->save();

        return back()->with('success', 'Domain verification code updated successfully');
    }

    public function storeTiktok(Request $request)
    {
        $request->validate([
            'pixel_id' => 'required|string',
            'access_token' => 'nullable|string',
            'test_event_code' => 'nullable|string',
        ]);

        if ($request->id) {
            $pixel = TiktokPixel::findOrFail($request->id);
        } else {
            $pixel = new TiktokPixel();
        }

        $pixel->pixel_id = $request->pixel_id;
        $pixel->access_token = $request->access_token;
        $pixel->test_event_code = $request->test_event_code;
        $pixel->is_active = $request->has('is_active');
        $pixel->save();

        return back()->with('success', 'TikTok Pixel saved successfully');
    }

    public function destroyTiktok(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:tiktok_pixels,id',
        ]);

        $pixel = TiktokPixel::findOrFail($request->id);
        $pixel->delete();

        return back()->with('success', 'TikTok Pixel deleted successfully');
    }

    public function storeGtm(Request $request)
    {
        $request->validate([
            'gtm_id' => 'required|string|unique:gtm_containers,gtm_id,' . $request->id,
        ]);

        if ($request->id) {
            $container = GtmContainer::findOrFail($request->id);
        } else {
            $container = new GtmContainer();
        }

        $container->gtm_id = $request->gtm_id;
        $container->is_active = $request->has('is_active');
        $container->save();

        return back()->with('success', 'GTM Container saved successfully');
    }

    public function destroyGtm(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:gtm_containers,id',
        ]);

        $container = GtmContainer::findOrFail($request->id);
        $container->delete();

        return back()->with('success', 'GTM Container deleted successfully');
    }

    public function store(Request $request)
    {
        $request->validate([
            'pixel_id' => 'required|string',
            'catalog_id' => 'nullable|string',
            'access_token' => 'nullable|string',
            'test_event_code' => 'nullable|string',
            'api_version' => 'nullable|string',
        ]);

        $id = $request->input('id');

        if ($id) {
            $pixel = FacebookPixel::findOrFail($id);
        } else {
            $pixel = new FacebookPixel();
        }

        $pixel->pixel_id = $request->pixel_id;
        $pixel->catalog_id = $request->catalog_id;
        $pixel->access_token = $request->access_token;
        $pixel->test_event_code = $request->test_event_code;
        $pixel->api_version = $request->api_version ?? 'v19.0';
        $pixel->is_active = $request->has('is_active');
        $pixel->server_side_enabled = $request->has('server_side_enabled');
        $pixel->save();

        return back()->with('success', 'Pixel configuration saved successfully');
    }

    public function destroy(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:facebook_pixels,id',
        ]);

        $pixel = FacebookPixel::findOrFail($request->id);
        $pixel->delete();
        
        return back()->with('success', 'Pixel configuration deleted successfully');
    }
}
