<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class OrderController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:orders.manage');
    }

    public function index(Request $request)
    {
        $status = $request->query('status');
        $tab = $request->query('tab', 'all');
        $query = Order::with(['user', 'items.product'])->latest();
        
        // Search Filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhereHas('user', function($u) use ($search) {
                      $u->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        // Date Filter
        $startDate = $request->input('start_date', now()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));

        $query->whereDate('created_at', '>=', $startDate)
              ->whereDate('created_at', '<=', $endDate);
        
        if ($status === 'incomplete') {
            // Incomplete Orders Page Logic
            switch ($tab) {
                case 'all':
                    $query->where(function($q) {
                        $q->where('status', 'incomplete')
                          ->orWhere('is_from_incomplete', true);
                    })->withTrashed();
                    break;
                case 'pending': // This corresponds to "New Orders" tab
                    $query->where(function($q) {
                        $q->where('status', 'incomplete')
                          ->orWhere(function($sub) {
                              $sub->where('is_from_incomplete', true)
                                  ->whereIn('status', ['pending', 'hold', 'unpaid']);
                          });
                    });
                    break;
                case 'processing': // This corresponds to "Confirmed" tab
                    $query->where('is_from_incomplete', true)
                          ->whereIn('status', ['processing', 'on_delivery', 'shipped', 'delivered', 'returned']);
                    break;
                case 'cancelled':
                    $query->where('is_from_incomplete', true)->where('status', 'cancelled');
                    break;
                case 'deleted':
                    $query->onlyTrashed()->where('is_from_incomplete', true);
                    break;
                default:
                    $query->where('status', 'incomplete');
            }
        } elseif ($status && $status !== 'all') {
            $query->where('status', $status);
        } else {
            // Exclude incomplete orders from 'all' view
            $query->where('status', '!=', 'incomplete');
        }
        
        $orders = $query->paginate(10);

        // Counts & Amounts
        $statuses = ['pending', 'processing', 'shipped', 'on_delivery', 'cancelled', 'hold', 'delivered', 'unpaid'];
        
        // Base query for counts
        $baseQuery = Order::query();
        $baseQuery->whereDate('created_at', '>=', $startDate)
                  ->whereDate('created_at', '<=', $endDate);
        if ($request->filled('search')) {
             $search = $request->search;
             $baseQuery->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhereHas('user', function($u) use ($search) {
                      $u->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        // 'All' count should also exclude incomplete
        $counts = ['all' => (clone $baseQuery)->where('status', '!=', 'incomplete')->count()];
        $amounts = ['all' => (clone $baseQuery)->where('status', '!=', 'incomplete')->sum('total_amount')];

        foreach ($statuses as $s) {
            $counts[$s] = (clone $baseQuery)->where('status', $s)->count();
            $amounts[$s] = (clone $baseQuery)->where('status', $s)->sum('total_amount');
        }

        // Incomplete Specific Counts
        if ($status === 'incomplete') {
            $counts['incomplete_all'] = (clone $baseQuery)->where(function($q) {
                $q->where('status', 'incomplete')
                  ->orWhere('is_from_incomplete', true);
            })->withTrashed()->count();
            
            $counts['incomplete_pending'] = (clone $baseQuery)->where(function($q) {
                $q->where('status', 'incomplete')
                  ->orWhere(function($sub) {
                      $sub->where('is_from_incomplete', true)
                          ->whereIn('status', ['pending', 'hold', 'unpaid']);
                  });
            })->count();
            $counts['incomplete_processing'] = (clone $baseQuery)->where('is_from_incomplete', true)
                ->whereIn('status', ['processing', 'on_delivery', 'shipped', 'delivered', 'returned'])
                ->count();
            $counts['incomplete_cancelled'] = (clone $baseQuery)->where('is_from_incomplete', true)->where('status', 'cancelled')->count();
            $counts['incomplete_deleted'] = Order::onlyTrashed()->where('is_from_incomplete', true)
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->count();
        }

        // Incomplete Count (for the badge)
        $incompleteCount = (clone $baseQuery)->where('status', 'incomplete')->count();

        // Placeholders
        $placeholders = ['ship_later', 'returned', 'lost', 'deleted'];
        foreach ($placeholders as $s) {
            if ($status !== 'incomplete' || $s !== 'deleted') {
                $counts[$s] = 0;
                $amounts[$s] = 0;
            }
        }

        return view('admin.orders.index', compact('orders', 'status', 'counts', 'amounts', 'startDate', 'endDate', 'incompleteCount', 'tab'));
    }

    public function show($id)
    {
        $order = Order::with(['items.product', 'user'])->findOrFail($id);
        return view('admin.orders.show', compact('order'));
    }

    public function update(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        
        $data = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'phone' => 'required|string|min:11|max:11',
            'email' => 'nullable|email|max:255',
            'address' => 'required|string',
            'city' => 'nullable|string',
            'delivery_area' => 'nullable|string',
            'district' => 'nullable|string',
            'thana' => 'nullable|string',
            'advance_amount' => 'nullable|numeric',
            'payment_sender_number' => 'nullable|string',
            'payment_receiver_number' => 'nullable|string',
            'note' => 'nullable|string',
            'staff_note' => 'nullable|string',
            'extra_charge' => 'nullable|numeric',
            'discount_amount' => 'nullable|numeric',
            'delivery_cost' => 'nullable|numeric',
            'status' => 'required|string',
            'payment_method' => 'nullable|string',
            'items' => 'nullable|array',
            'attachment' => 'nullable|file|max:10240', // Max 10MB
        ]);

        if ($order->status === 'incomplete' && $data['status'] !== 'incomplete') {
            $data['is_from_incomplete'] = true;
        }

        // Handle File Upload
        if ($request->hasFile('attachment')) {
            // Delete old file if exists
            if ($order->attachment) {
                Storage::disk('public')->delete($order->attachment);
            }
            $path = $request->file('attachment')->store('attachments', 'public');
            $data['attachment'] = $path;
        }

        $order->update($data);

        // Update Items
        if ($request->has('items')) {
            foreach ($request->items as $itemId => $itemData) {
                $item = \App\Models\OrderItem::find($itemId);
                if ($item && $item->order_id == $order->id) {
                    $item->update([
                        'quantity' => $itemData['quantity'],
                        'price' => $itemData['price'],
                    ]);
                }
            }
        }

        // Add New Items
        if ($request->has('new_items')) {
            foreach ($request->new_items as $newItem) {
                $product = \App\Models\Product::find($newItem['product_id']);
                if ($product) {
                    $order->items()->create([
                        'product_id' => $product->id,
                        'product_name' => $product->name,
                        'quantity' => $newItem['quantity'],
                        'price' => $newItem['price'] ?? $product->price,
                    ]);
                }
            }
        }

        // Recalculate order total
        $totalAmount = $order->items()->get()->sum(function($item) {
            return $item->price * $item->quantity;
        });
        
        $deliveryCost = $request->input('delivery_cost', $order->delivery_cost);
        $discountAmount = $request->input('discount_amount', $order->discount_amount);
        $extraCharge = $request->input('extra_charge', $order->extra_charge);
        
        // Final Total Calculation logic might be complex depending on business rules
        // For now just updating total_amount based on items sum, assuming other fields handled by $order->update($data)
        // But wait, $data includes delivery_cost etc.
        // Let's ensure total_amount is consistent
        // Usually total_amount = items_total + delivery + extra - discount - advance? 
        // Or just the invoice total. 
        // Existing code was just summing items. Let's keep it simple and just sum items for now, 
        // or rely on what was there. The previous code was:
        // $order->update(['total_amount' => $totalAmount]);
        // I'll stick to that.

        $order->update(['total_amount' => $totalAmount]);

        if ($request->input('redirect_to') === 'incomplete') {
            return redirect()->route('admin.orders.index', ['status' => 'incomplete'])->with('success', 'Order updated successfully.');
        } elseif ($request->input('redirect_to') === 'complete') {
            return redirect()->route('admin.orders.index')->with('success', 'Order updated successfully.');
        }

        return back()->with('success', 'Order updated successfully.');
    }

    public function updateStatus(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        $request->validate([
            'status' => 'required|string',
        ]);
        
        $oldStatus = $order->status;
        $newStatus = $request->status;

        $updateData = ['status' => $newStatus];

        if ($oldStatus === 'incomplete' && $newStatus !== 'incomplete') {
            $updateData['is_from_incomplete'] = true;
        }
        
        $order->update($updateData);
        
        return back()->with('success', 'Order status updated successfully.');
    }

    public function destroy($id)
    {
        $order = Order::findOrFail($id);
        $order->delete();
        
        return back()->with('success', 'Order deleted successfully.');
    }
}
