<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\LandingPage;
use App\Models\Product;

class LandingPageController extends Controller
{
    public function index()
    {
        $landingPages = LandingPage::with('product')->latest()->paginate(10);
        return view('admin.landing_pages.index', compact('landingPages'));
    }

    public function create()
    {
        $products = Product::select('id', 'name')->get();
        return view('admin.landing_pages.create', compact('products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:landing_pages',
            'product_id' => 'required|exists:products,id',
            'status' => 'required|in:active,inactive',
        ]);

        LandingPage::create($request->all());

        return redirect()->route('admin.landing-pages.index')->with('success', 'Landing page created successfully');
    }

    public function campaign()
    {
        return view('admin.landing_pages.campaign');
    }

    public function edit($id)
    {
        $landingPage = LandingPage::findOrFail($id);
        $products = Product::select('id', 'name')->get();
        return view('admin.landing_pages.edit', compact('landingPage', 'products'));
    }

    public function update(Request $request, $id)
    {
        $landingPage = LandingPage::findOrFail($id);
        
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:landing_pages,slug,' . $id,
            'product_id' => 'required|exists:products,id',
            'status' => 'required|in:active,inactive',
            'is_home' => 'boolean',
        ]);

        if ($request->is_home) {
            LandingPage::where('id', '!=', $id)->update(['is_home' => false]);
        }

        $landingPage->update($request->all());

        return redirect()->route('admin.landing-pages.index')->with('success', 'Landing page updated successfully');
    }

    public function builder($id)
    {
        $landingPage = LandingPage::findOrFail($id);

        // If content is empty (e.g. newly created page), load the default template
        if (empty($landingPage->content)) {
            $landingPage->content = view('admin.landing_pages.template', compact('landingPage'))->render();
        }

        return view('admin.landing_pages.builder', compact('landingPage'));
    }

    public function updateContent(Request $request, $id)
    {
        $landingPage = LandingPage::findOrFail($id);
        
        $request->validate([
            'content' => 'nullable|string',
            'css' => 'nullable|string',
        ]);

        $landingPage->update([
            'content' => $request->content,
            'css' => $request->css,
        ]);

        return response()->json(['success' => true]);
    }

    public function destroy($id)
    {
        $landingPage = LandingPage::findOrFail($id);
        $landingPage->delete();
        
        return back()->with('success', 'Landing page deleted successfully');
    }
}
