<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Inventory;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InventoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::select('id', 'name', 'thumbnail', 'sku', 'stock_quantity', 'updated_at')
                           ->latest('updated_at')
                           ->paginate(10);
        return view('admin.inventory.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $products = Product::select('id', 'name')->orderBy('name')->get();
        $selectedProductId = $request->query('product_id');
        return view('admin.inventory.create', compact('products', 'selectedProductId'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'type' => 'required|in:in,out',
            'reason' => 'nullable|string|max:255',
        ]);

        $data = $request->all();
        $data['user_id'] = Auth::id();

        // Update product stock (assuming Product model has stock field or method)
        $product = Product::findOrFail($request->product_id);
        if ($request->type === 'in') {
            $product->increment('stock_quantity', $request->quantity);
        } else {
            if ($product->stock_quantity < $request->quantity) {
                return back()->withErrors(['quantity' => 'Insufficient stock for this operation.']);
            }
            $product->decrement('stock_quantity', $request->quantity);
        }

        Inventory::create($data);

        return redirect()->route('admin.inventory.index')->with('success', 'Inventory adjustment recorded successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        // Inventory logs should generally not be editable to maintain audit trail
        // But if needed, can be implemented
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
