<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Product;
use App\Models\OrderItem;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // Default to Today if no date range is selected
        $startDate = $request->input('start_date', date('Y-m-d'));
        $endDate = $request->input('end_date', date('Y-m-d'));

        // Base Query for Orders
        $ordersQuery = Order::query()->where('status', '!=', 'incomplete');

        if ($startDate && $endDate) {
            $ordersQuery->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        }

        // 1. All Orders
        $allOrdersCount = (clone $ordersQuery)->count();
        $allOrdersValue = (clone $ordersQuery)->sum('total_amount');

        // 2. Cancelled Orders
        $cancelledOrdersCount = (clone $ordersQuery)->where('status', 'cancelled')->count();
        $cancelledOrdersValue = (clone $ordersQuery)->where('status', 'cancelled')->sum('total_amount');

        // 3. Delivered Orders
        $deliveredOrdersCount = (clone $ordersQuery)->where('status', 'delivered')->count();
        $deliveredOrdersValue = (clone $ordersQuery)->where('status', 'delivered')->sum('total_amount');

        // 4. Return Orders (Assuming 'returned' status exists or using 0 if not used yet)
        $returnedOrdersCount = (clone $ordersQuery)->where('status', 'returned')->count();
        $returnedOrdersValue = (clone $ordersQuery)->where('status', 'returned')->sum('total_amount');

        // 5. Hold Orders
        $holdOrdersCount = (clone $ordersQuery)->where('status', 'hold')->count();
        $holdOrdersValue = (clone $ordersQuery)->where('status', 'hold')->sum('total_amount');

        // 6. Confirmed Orders (Mapping 'processing' to Confirmed for now)
        $confirmedOrdersCount = (clone $ordersQuery)->whereIn('status', ['processing', 'confirmed'])->count();
        $confirmedOrdersValue = (clone $ordersQuery)->whereIn('status', ['processing', 'confirmed'])->sum('total_amount');

        // 7. Profit/Loss Calculation
        // Revenue = Delivered + Confirmed + Processing + Shipped
        $revenue = (clone $ordersQuery)->whereIn('status', ['delivered', 'confirmed', 'processing', 'shipped'])->sum('total_amount');
        
        // Product Costs: sum(quantity * product.purchase_price) for sold items within date range
        $productCostsQuery = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->where('orders.status', '!=', 'incomplete')
            ->whereIn('orders.status', ['delivered', 'confirmed', 'processing', 'shipped']);
        if ($startDate && $endDate) {
            $productCostsQuery->whereBetween('orders.created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        }
        $productCosts = (float) $productCostsQuery->sum(DB::raw('order_items.quantity * COALESCE(products.purchase_price, 0)'));

        $profit = $revenue - $productCosts;



        // Orders by Status
        $ordersByStatus = (clone $ordersQuery)->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get();

        // Product Reports
        $productReports = Product::select('id', 'name', 'thumbnail')
            ->whereHas('orderItems', function ($query) use ($startDate, $endDate) {
                $query->whereHas('order', function ($q) use ($startDate, $endDate) {
                    $q->where('status', '!=', 'incomplete');
                    if ($startDate && $endDate) {
                        $q->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
                    }
                });
            })
            ->withCount(['orderItems as total_orders' => function ($query) use ($startDate, $endDate) {
                $query->whereHas('order', function ($q) use ($startDate, $endDate) {
                    $q->where('status', '!=', 'incomplete');
                    if ($startDate && $endDate) {
                        $q->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
                    }
                });
            }])
            ->addSelect(['total_order_value' => OrderItem::selectRaw('sum(quantity * price)')
                ->whereColumn('product_id', 'products.id')
                ->whereHas('order', function ($q) use ($startDate, $endDate) {
                    $q->where('status', '!=', 'incomplete');
                    if ($startDate && $endDate) {
                        $q->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
                    }
                })
            ])
            ->withCount(['orderItems as cancelled_orders' => function ($query) use ($startDate, $endDate) {
                $query->whereHas('order', function ($q) use ($startDate, $endDate) {
                    $q->where('status', 'cancelled');
                    if ($startDate && $endDate) {
                        $q->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
                    }
                });
            }])
            ->withCount(['orderItems as pending_orders' => function ($query) use ($startDate, $endDate) {
                $query->whereHas('order', function ($q) use ($startDate, $endDate) {
                    $q->where('status', 'pending');
                    if ($startDate && $endDate) {
                        $q->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
                    }
                });
            }])
            ->orderByDesc('total_orders')
            ->take(10)
            ->get();

        // Charts Data (Last 30 Days)
        $dates = collect();
        for ($i = 29; $i >= 0; $i--) {
            $dates->push(now()->subDays($i)->format('Y-m-d'));
        }

        $chartOrders = Order::where('status', '!=', 'incomplete')
            ->where('created_at', '>=', now()->subDays(30))
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('count(*) as count'))
            ->groupBy('date')
            ->pluck('count', 'date');

        $chartEarnings = Order::where('status', '!=', 'incomplete')
            ->whereIn('status', ['delivered', 'confirmed', 'processing', 'shipped']) // Included shipped, removed completed
            ->where('created_at', '>=', now()->subDays(30))
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('sum(total_amount) as total'))
            ->groupBy('date')
            ->pluck('total', 'date');

        $chartLabels = $dates;
        
        // FIX: Provide both Singular and Plural variable names to support old and new views
        $chartOrdersData = $dates->map(fn($date) => $chartOrders[$date] ?? 0);
        $chartEarningsData = $dates->map(fn($date) => $chartEarnings[$date] ?? 0);
        
        $chartOrderData = $chartOrdersData; // Alias
        $chartEarningData = $chartEarningsData; // Alias

        // Monthly Comparison (This Month vs Last Month)
        $currentMonthStart = now()->startOfMonth();
        $currentMonthEnd = now()->endOfMonth();
        $lastMonthStart = now()->subMonth()->startOfMonth();
        $lastMonthEnd = now()->subMonth()->endOfMonth();

        // 1. Earnings Growth (Delivered + Confirmed + Processing + Shipped)
        $currentMonthEarnings = Order::where('status', '!=', 'incomplete')
            ->whereIn('status', ['delivered', 'confirmed', 'processing', 'shipped'])
            ->whereBetween('created_at', [$currentMonthStart, $currentMonthEnd])
            ->sum('total_amount');

        $lastMonthEarnings = Order::where('status', '!=', 'incomplete')
            ->whereIn('status', ['delivered', 'confirmed', 'processing', 'shipped'])
            ->whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])
            ->sum('total_amount');

        $earningsPercentage = 0;
        if ($lastMonthEarnings > 0) {
            $earningsPercentage = (($currentMonthEarnings - $lastMonthEarnings) / $lastMonthEarnings) * 100;
        } elseif ($currentMonthEarnings > 0) {
            $earningsPercentage = 100;
        }

        return view('admin.dashboard', compact(
            'allOrdersCount', 'allOrdersValue',
            'cancelledOrdersCount', 'cancelledOrdersValue',
            'deliveredOrdersCount', 'deliveredOrdersValue',
            'returnedOrdersCount', 'returnedOrdersValue',
            'holdOrdersCount', 'holdOrdersValue',
            'confirmedOrdersCount', 'confirmedOrdersValue',
            'ordersByStatus',
            'productReports',
            'startDate',
            'endDate',
            'chartLabels',
            'chartOrdersData', // Fix: Plural
            'chartEarningsData', // Fix: Plural
            'chartOrderData', // Alias
            'chartEarningData', // Alias
            'earningsPercentage',
            'revenue',
            'productCosts',
            'profit'
        ));
    }
}
