<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:categories.manage');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $type = $request->query('type', 'main'); // main, sub, child
        
        $query = Category::query();

        if ($type === 'main') {
            $query->whereNull('parent_id');
        } elseif ($type === 'sub') {
            $query->whereNotNull('parent_id')
                  ->whereHas('parent', function($q) {
                      $q->whereNull('parent_id');
                  });
        } elseif ($type === 'child') {
            $query->whereNotNull('parent_id')
                  ->whereHas('parent', function($q) {
                      $q->whereNotNull('parent_id');
                  });
        }

        $categories = $query->latest()->paginate(10);

        return view('admin.categories.index', compact('categories', 'type'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $type = $request->query('type', 'main');

        // Fetch all potential parents grouped by type
        $mainCategories = Category::whereNull('parent_id')->get();
        
        $subCategories = Category::whereNotNull('parent_id')
            ->whereHas('parent', function($q) {
                $q->whereNull('parent_id');
            })->get();

        return view('admin.categories.create', compact('type', 'mainCategories', 'subCategories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:categories,id',
            'image' => 'nullable|image|max:2048',
            'banner_image' => 'nullable|image|max:4096',
            'home_banner_image' => 'nullable|image|max:4096',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'status' => 'boolean',
        ]);

        $data = $request->except(['image', 'banner_image', 'home_banner_image']);
        $data['slug'] = Str::slug($request->name);

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('categories', 'public');
        }

        if ($request->hasFile('banner_image')) {
            $data['banner_image'] = $request->file('banner_image')->store('categories/banners', 'public');
        }

        if ($request->hasFile('home_banner_image')) {
            $data['home_banner_image'] = $request->file('home_banner_image')->store('categories/home_banners', 'public');
        }

        Category::create($data);

        $type = 'main';
        if ($request->parent_id) {
            $parent = Category::find($request->parent_id);
            if ($parent && $parent->parent_id) {
                $type = 'child';
            } else {
                $type = 'sub';
            }
        }

        return redirect()->route('admin.categories.index', ['type' => $type])
            ->with('success', 'Category created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $category = Category::findOrFail($id);
        
        // Determine type
        $type = 'main';
        if ($category->parent_id) {
            if ($category->parent->parent_id) {
                $type = 'child';
            } else {
                $type = 'sub';
            }
        }

        $parents = collect();
        if ($type === 'sub') {
            $parents = Category::whereNull('parent_id')->where('id', '!=', $id)->get();
        } elseif ($type === 'child') {
            $parents = Category::whereNotNull('parent_id')
                ->whereHas('parent', function($q) {
                    $q->whereNull('parent_id');
                })->where('id', '!=', $id)->get();
        }

        return view('admin.categories.edit', compact('category', 'type', 'parents'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $category = Category::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:categories,id',
            'image' => 'nullable|image|max:2048',
            'banner_image' => 'nullable|image|max:4096',
            'home_banner_image' => 'nullable|image|max:4096',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'status' => 'boolean',
        ]);

        $data = $request->except(['image', 'banner_image', 'home_banner_image']);
        $data['slug'] = Str::slug($request->name);

        if ($request->hasFile('image')) {
            // Delete old image
            if ($category->image) {
                Storage::disk('public')->delete($category->image);
            }
            $data['image'] = $request->file('image')->store('categories', 'public');
        }

        if ($request->hasFile('banner_image')) {
            // Delete old banner_image
            if ($category->banner_image) {
                Storage::disk('public')->delete($category->banner_image);
            }
            $data['banner_image'] = $request->file('banner_image')->store('categories/banners', 'public');
        }

        if ($request->hasFile('home_banner_image')) {
            // Delete old home_banner_image
            if ($category->home_banner_image) {
                Storage::disk('public')->delete($category->home_banner_image);
            }
            $data['home_banner_image'] = $request->file('home_banner_image')->store('categories/home_banners', 'public');
        }

        $category->update($data);

        // Redirect based on type
        $type = 'main';
        if ($category->parent_id) {
            $parent = Category::find($category->parent_id); // Use find to be safe
            if ($parent && $parent->parent_id) {
                $type = 'child';
            } else {
                $type = 'sub';
            }
        }

        return redirect()->route('admin.categories.index', ['type' => $type])
            ->with('success', 'Category updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $category = Category::findOrFail($id);
        
        // Determine type for redirect
        $type = 'main';
        if ($category->parent_id) {
            if ($category->parent->parent_id) {
                $type = 'child';
            } else {
                $type = 'sub';
            }
        }

        if ($category->image) {
            Storage::disk('public')->delete($category->image);
        }

        $category->delete();

        return redirect()->route('admin.categories.index', ['type' => $type])
            ->with('success', 'Category deleted successfully.');
    }
}
