<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\BannerCategory;
use Illuminate\Support\Str;

use App\Models\Banner;
use App\Models\Category; // Product Category
use Illuminate\Support\Facades\Storage;

class BannerController extends Controller
{
    public function index()
    {
        $banners = Banner::with('category')->latest()->get();
        return view('admin.banners.index', compact('banners'));
    }

    public function category()
    {
        $categories = BannerCategory::latest()->get();
        return view('admin.banners.category', compact('categories'));
    }

    public function storeCategory(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:banner_categories,slug,' . $request->id,
        ]);

        if ($request->id) {
            $category = BannerCategory::findOrFail($request->id);
        } else {
            $category = new BannerCategory();
        }

        $category->name = $request->name;
        $category->slug = $request->slug ? Str::slug($request->slug) : Str::slug($request->name);
        $category->status = $request->has('status');
        $category->save();

        return back()->with('success', 'Banner Category saved successfully');
    }

    public function destroyCategory(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:banner_categories,id',
        ]);

        $category = BannerCategory::findOrFail($request->id);
        $category->delete();

        return back()->with('success', 'Banner Category deleted successfully');
    }

    public function create()
    {
        $categories = BannerCategory::where('status', true)->get();
        $productCategories = Category::all(); // Fetch all product categories
        return view('admin.banners.create', compact('categories', 'productCategories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'nullable|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'text_position' => 'required|in:left,center,right',
            'banner_category_id' => 'nullable|exists:banner_categories,id',
            'product_category_id' => 'nullable|exists:categories,id',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'url' => 'nullable|string|max:255',
        ]);

        $banner = new Banner();
        $banner->title = $request->title;
        $banner->subtitle = $request->subtitle;
        $banner->text_position = $request->text_position;
        $banner->banner_category_id = $request->banner_category_id;
        $banner->product_category_id = $request->product_category_id;
        $banner->url = $request->url;
        $banner->status = $request->has('status');

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('banners', 'public');
            $banner->image = $path;
        }

        $banner->save();

        return redirect()->route('admin.banners.index')->with('success', 'Banner created successfully');
    }

    public function edit($id)
    {
        $banner = Banner::findOrFail($id);
        $categories = BannerCategory::where('status', true)->get();
        $productCategories = Category::all(); // Fetch all product categories
        return view('admin.banners.edit', compact('banner', 'categories', 'productCategories'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'nullable|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'text_position' => 'required|in:left,center,right',
            'banner_category_id' => 'nullable|exists:banner_categories,id',
            'product_category_id' => 'nullable|exists:categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'url' => 'nullable|string|max:255',
        ]);

        $banner = Banner::findOrFail($id);
        $banner->title = $request->title;
        $banner->subtitle = $request->subtitle;
        $banner->text_position = $request->text_position;
        $banner->banner_category_id = $request->banner_category_id;
        $banner->product_category_id = $request->product_category_id;
        $banner->url = $request->url;
        $banner->status = $request->has('status');

        if ($request->hasFile('image')) {
            if ($banner->image) {
                Storage::disk('public')->delete($banner->image);
            }
            $path = $request->file('image')->store('banners', 'public');
            $banner->image = $path;
        }

        $banner->save();

        return redirect()->route('admin.banners.index')->with('success', 'Banner updated successfully');
    }

    public function destroy($id)
    {
        $banner = Banner::findOrFail($id);
        
        if ($banner->image && Storage::disk('public')->exists($banner->image)) {
            Storage::disk('public')->delete($banner->image);
        }
        
        $banner->delete();

        return back()->with('success', 'Banner deleted successfully');
    }
}
