<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Models\Order;
use Illuminate\Support\Facades\Http;
use App\Services\SteadfastService;

class ApiController extends Controller
{
    public function payment()
    {
        $setting = GeneralSetting::first();
        return view('admin.api.payment', compact('setting'));
    }

    public function updatePayment(Request $request)
    {
        $request->validate([
            'bkash_app_key' => 'nullable|string',
            'bkash_app_secret' => 'nullable|string',
            'bkash_username' => 'nullable|string',
            'bkash_password' => 'nullable|string',
            'sslcz_store_id' => 'nullable|string',
            'sslcz_store_password' => 'nullable|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }

        // bKash Settings
        $setting->bkash_enabled = $request->has('bkash_enabled');
        $setting->bkash_app_key = $request->bkash_app_key;
        $setting->bkash_app_secret = $request->bkash_app_secret;
        $setting->bkash_username = $request->bkash_username;
        $setting->bkash_password = $request->bkash_password;
        $setting->bkash_test_mode = $request->has('bkash_test_mode');

        // SSLCommerz Settings
        $setting->sslcz_enabled = $request->has('sslcz_enabled');
        $setting->sslcz_store_id = $request->sslcz_store_id;
        $setting->sslcz_store_password = $request->sslcz_store_password;
        $setting->sslcz_test_mode = $request->has('sslcz_test_mode');

        $setting->save();

        return back()->with('success', 'Payment Gateway settings updated successfully.');
    }

    public function sms()
    {
        $setting = GeneralSetting::first();
        return view('admin.api.sms', compact('setting'));
    }

    public function updateSms(Request $request)
    {
        $request->validate([
            'sms_api_key' => 'required|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }
        $setting->sms_api_key = $request->sms_api_key;
        $setting->sms_active = $request->has('sms_active');
        $setting->sms_order_confirm = $request->has('sms_order_confirm');
        $setting->sms_forgot_password = $request->has('sms_forgot_password');
        $setting->sms_password_generator = $request->has('sms_password_generator');
        $setting->save();

        return back()->with('success', 'SMS API settings updated successfully.');
    }

    public function testSms(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'message' => 'required|string|max:160',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting || !$setting->sms_api_key) {
            return back()->with('error', 'Please configure SMS API Key first.');
        }

        $response = Http::get('https://fraudchecker.link/api/v1/sms/', [
            'api_key' => $setting->sms_api_key,
            'number' => $request->phone,
            'message' => $request->message,
        ]);

        $data = $response->json();

        if (isset($data['status']) && $data['status'] == 'success') {
            return back()->with('sms_result', $data);
        } else {
             $errorMsg = $data['message'] ?? 'Unknown Error';
             return back()->with('error', 'SMS Error: ' . $errorMsg);
        }
    }

    public function checkSmsBalance()
    {
         $setting = GeneralSetting::first();
         if (!$setting || !$setting->sms_api_key) {
             return response()->json(['status' => 'error', 'message' => 'API Key not configured']);
         }

         $response = Http::get('https://fraudchecker.link/api/v1/sms/balance.php', [
             'api_key' => $setting->sms_api_key,
         ]);
         
         return $response->json();
    }

    public function autoCall()
    {
        $setting = GeneralSetting::first();
        return view('admin.api.auto_call', compact('setting'));
    }

    public function updateAutoCall(Request $request)
    {
        $request->validate([
            'twilio_account_sid' => 'nullable|string',
            'twilio_auth_token' => 'nullable|string',
            'twilio_phone_number' => 'nullable|string',
            'call_center_number' => 'nullable|string',
        ]);

        $setting = GeneralSetting::first() ?? new GeneralSetting();
        $setting->auto_call_enabled = $request->has('auto_call_enabled');
        $setting->twilio_account_sid = $request->twilio_account_sid;
        $setting->twilio_auth_token = $request->twilio_auth_token;
        $setting->twilio_phone_number = $request->twilio_phone_number;
        $setting->call_center_number = $request->call_center_number;
        $setting->save();

        return back()->with('success', 'Auto Call settings updated successfully.');
    }

    public function testAutoCall(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'phone' => 'required|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting || !$setting->twilio_account_sid || !$setting->twilio_auth_token || !$setting->twilio_phone_number) {
            return back()->with('error', 'Please configure Twilio credentials first.');
        }

        $order = \App\Models\Order::findOrFail($request->order_id);
        $order->update(['call_status' => 'CALL_QUEUED']);

        $url = route('voice.start', ['order' => $order->id]);
        $statusCallback = route('voice.status', ['order_id' => $order->id]);

        try {
            $response = Http::withBasicAuth($setting->twilio_account_sid, $setting->twilio_auth_token)
                ->asForm()
                ->post("https://api.twilio.com/2010-04-01/Accounts/{$setting->twilio_account_sid}/Calls.json", [
                    'To' => $request->phone,
                    'From' => $setting->twilio_phone_number,
                    'Url' => $url,
                    'Method' => 'POST',
                    'StatusCallback' => $statusCallback,
                    'StatusCallbackMethod' => 'POST',
                    'StatusCallbackEvent' => ['initiated', 'ringing', 'answered', 'completed'],
                ]);

            if ($response->successful()) {
                $data = $response->json();
                \App\Models\OrderCall::create([
                    'order_id' => $order->id,
                    'call_sid' => $data['sid'] ?? null,
                    'attempt' => 1,
                    'call_result' => 'queued',
                ]);
                $order->update(['call_status' => 'CALL_INITIATED']);
                return back()->with('success', 'Test call initiated. Twilio SID: ' . ($data['sid'] ?? 'n/a'));
            }

            return back()->with('error', 'Twilio API Error: ' . $response->body());
        } catch (\Throwable $e) {
            return back()->with('error', 'Request failed: ' . $e->getMessage());
        }
    }

    public function courier(SteadfastService $steadfastService)
    {
        $setting = GeneralSetting::first();
        
        // Steadfast Balance
        $steadfastBalance = $steadfastService->getBalance();

        // Pending Deliveries (All active orders that are not delivered/cancelled/returned)
        $pendingDeliveries = Order::whereNotIn('status', ['delivered', 'cancelled', 'returned'])->count();

        // Order Stats Breakdown
        $orderStats = [
            'pending' => Order::where('status', 'pending')->count(),
            'delivered' => Order::where('status', 'delivered')->count(),
            'cancelled' => Order::where('status', 'cancelled')->count(),
        ];

        return view('admin.api.courier', compact('setting', 'steadfastBalance', 'pendingDeliveries', 'orderStats'));
    }

    public function sendOrderToSteadfast($id, SteadfastService $steadfastService)
    {
        $order = Order::findOrFail($id);

        if ($order->courier_tracking_id) {
            return back()->with('error', 'Order already sent to courier. Tracking ID: ' . $order->courier_tracking_id);
        }

        // Restriction: Only shipped (Ready to Delivery) orders
        if ($order->status !== 'shipped') {
            return back()->with('error', 'Order must be in "Ready to Delivery" status to send to Steadfast.');
        }

        $result = $steadfastService->createOrder($order);

        if ($result['status'] === 'success') {
            $order->update([
                'courier_name' => 'Steadfast',
                'courier_consignment_id' => $result['consignment_id'],
                'courier_tracking_id' => $result['tracking_code'],
                'courier_status' => $result['courier_status'],
            ]);

            return back()->with('success', 'Order sent to Steadfast successfully. Tracking ID: ' . $result['consignment_id']);
        } else {
            return back()->with('error', 'Failed to send to Steadfast: ' . $result['message']);
        }
    }

    public function checkCourierStatus($id, SteadfastService $steadfastService)
    {
        $order = Order::findOrFail($id);

        if (!$order->courier_tracking_id) {
            return back()->with('error', 'This order has not been sent to any courier yet.');
        }

        $result = $steadfastService->checkStatus($order->courier_tracking_id);

        $updateData = [];

        // Try to update consignment_id if available and missing
        if (isset($result['consignment_id']) && !$order->courier_consignment_id) {
            $updateData['courier_consignment_id'] = $result['consignment_id'];
        }

        if (isset($result['delivery_status']) || isset($result['status'])) {
             $status = $result['delivery_status'] ?? $result['status'];
             $updateData['courier_status'] = $status;
             
             // Auto-update local order status based on courier status
             $statusLower = strtolower($status);
             if ($statusLower === 'delivered') {
                 $updateData['status'] = 'delivered';
             } elseif (str_contains($statusLower, 'return')) {
                 $updateData['status'] = 'returned';
             }
             
             $order->update($updateData);
             return back()->with('success', 'Courier status updated: ' . $status);
        }
        
        return back()->with('error', 'Failed to fetch status: ' . ($result['message'] ?? 'Unknown error'));
    }

    public function bulkAction(Request $request, SteadfastService $steadfastService)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:orders,id',
            'courier' => 'nullable|string',
            'status' => 'nullable|string',
        ]);

        $ids = $request->ids;
        $courier = $request->courier;
        $status = $request->status;
        
        // Handle Status Update
        if ($status) {
             // Validate same status constraint
             $firstOrder = Order::find($ids[0]);
             if (!$firstOrder) {
                 return back()->with('error', 'Order not found.');
             }
             $currentStatus = $firstOrder->status;
             
             foreach ($ids as $id) {
                 $order = Order::find($id);
                 if (!$order || $order->status !== $currentStatus) {
                      return back()->with('error', 'All selected orders must have the same status to perform this action.');
                 }
             }

             Order::whereIn('id', $ids)->update(['status' => $status]);
             return back()->with('success', 'Status updated successfully to ' . ucfirst($status));
        }

        if (!$courier) {
            return back()->with('error', 'Please select an action (Courier or Status).');
        }

        $results = [];

        if ($courier === 'check_status') {
            foreach ($ids as $id) {
                $order = Order::find($id);
                if (!$order) {
                     $results[] = [
                         'order_number' => 'ID: ' . $id,
                         'status' => 'error', 
                         'message' => 'Order not found'
                     ];
                     continue;
                }
                
                if (!$order->courier_tracking_id) {
                     $results[] = [
                         'order_number' => $order->order_number,
                         'status' => 'error', 
                         'message' => 'Not sent to courier'
                     ];
                     continue;
                }
                
                $result = $steadfastService->checkStatus($order->courier_tracking_id);
                
                if (isset($result['delivery_status']) || isset($result['status'])) {
                     $status = $result['delivery_status'] ?? $result['status'];
                     $updateData = ['courier_status' => $status];
                     
                     $statusLower = strtolower($status);
                     if ($statusLower === 'delivered') {
                         $updateData['status'] = 'delivered';
                     } elseif (str_contains($statusLower, 'return')) {
                         $updateData['status'] = 'returned';
                     }
                     
                     $order->update($updateData);
                     
                     $results[] = [
                         'order_number' => $order->order_number,
                         'status' => 'success',
                         'message' => 'Updated status to: ' . $status
                     ];
                } else {
                     $results[] = [
                         'order_number' => $order->order_number,
                         'status' => 'error', 
                         'message' => 'Failed: ' . ($result['message'] ?? 'Unknown error')
                     ];
                }
            }
            return back()->with('bulk_results', $results);
        }

        if ($courier === 'steadfast') {
            foreach ($ids as $id) {
                $order = Order::find($id);
                
                if (!$order) {
                     $results[] = [
                         'order_number' => 'ID: ' . $id,
                         'status' => 'error', 
                         'message' => 'Order not found'
                     ];
                     continue;
                }

                // Skip if already sent
                if ($order->courier_tracking_id) {
                     $results[] = [
                         'order_number' => $order->order_number,
                         'status' => 'error', 
                         'message' => 'Already sent (Tracking: ' . $order->courier_tracking_id . ')'
                     ];
                     continue;
                }

                // Restriction: Only shipped (Ready to Delivery) orders
                if ($order->status !== 'shipped') {
                     $results[] = [
                         'order_number' => $order->order_number,
                         'status' => 'error', 
                         'message' => 'Status is "' . $order->status . '". Must be "Ready to Delivery".'
                     ];
                     continue;
                }

                $result = $steadfastService->createOrder($order);

                if ($result['status'] === 'success') {
                    $order->update([
                        'courier_name' => 'Steadfast',
                        'courier_consignment_id' => $result['consignment_id'],
                        'courier_tracking_id' => $result['tracking_code'],
                        'courier_status' => $result['courier_status'],
                    ]);
                    $results[] = [
                        'order_number' => $order->order_number,
                        'status' => 'success',
                        'message' => 'Sent successfully. Tracking ID: ' . $result['consignment_id']
                    ];
                } else {
                    $results[] = [
                        'order_number' => $order->order_number,
                        'status' => 'error',
                        'message' => 'API Error: ' . ($result['message'] ?? 'Unknown Error')
                    ];
                }
            }
            
            return back()->with('bulk_results', $results);
        }

        return back()->with('error', 'Invalid courier selected.');
    }


    public function updateCourier(Request $request)
    {
        $request->validate([
            'courier_steadfast_api_key' => 'nullable|string',
            'courier_steadfast_secret_key' => 'nullable|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }
        
        $setting->courier_steadfast_enabled = $request->has('courier_steadfast_enabled');
        $setting->courier_steadfast_api_key = $request->courier_steadfast_api_key;
        $setting->courier_steadfast_secret_key = $request->courier_steadfast_secret_key;
        $setting->save();

        return back()->with('success', 'Courier API settings updated successfully.');
    }

    public function fraud()
    {
        $setting = GeneralSetting::first();
        return view('admin.api.fraud', compact('setting'));
    }

    public function updateFraud(Request $request)
    {
        $request->validate([
            'fraud_api_key' => 'required|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting) {
            $setting = new GeneralSetting();
        }
        $setting->fraud_api_key = $request->fraud_api_key;
        $setting->save();

        return back()->with('success', 'Fraud API settings updated successfully.');
    }

    public function checkFraud(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
        ]);

        $setting = GeneralSetting::first();
        if (!$setting || !$setting->fraud_api_key) {
            return back()->with('error', 'Please configure Fraud API Key first.');
        }

        $response = Http::asForm()->withHeaders([
            'Authorization' => 'Bearer ' . $setting->fraud_api_key,
        ])->post('https://fraudchecker.link/api/v1/qc/', [
            'phone' => $request->phone,
        ]);

        if ($response->successful()) {
                return back()->with('fraud_result', $response->json());
            } else {
                return back()->with('error', 'API Error: ' . $response->body());
            }
        }

        public function checkFraudJson(Request $request, \App\Services\FraudCheckService $fraudCheckService)
    {
        \Log::info('checkFraudJson called', $request->all());

        $request->validate([
            'phone' => 'required|string',
        ]);

        // If skip_external is true, try to find existing data first
        if ($request->boolean('skip_external')) {
            $existingOrder = Order::where('phone', $request->phone)
                ->whereNotNull('fraud_score')
                ->latest('fraud_checked_at')
                ->first();
            
            if ($existingOrder) {
                \Log::info('Found existing fraud data for phone: ' . $request->phone);
                $result = [
                    'success' => true,
                    'data' => $existingOrder->fraud_data,
                    'percentage' => $existingOrder->fraud_score
                ];
            } else {
                \Log::info('No existing fraud data found for phone: ' . $request->phone . ', skip_external is true, returning local data only');
                // Do not fallback to API if skip_external is requested and we want to avoid limits
                // Just return null data but success=true so we can show local orders
                $result = [
                    'success' => true,
                    'data' => null,
                    'percentage' => 0,
                    'message' => 'No external fraud data found, showing local history only.'
                ];
            }
        } else {
            \Log::info('skip_external is false, calling API for phone: ' . $request->phone);
            // Always fetch fresh data
            $result = $fraudCheckService->checkAndSave($request->phone);
        }

        \Log::info('Fraud check result', ['success' => $result['success'] ?? false]);

        // Fetch local order history (kept for display in modal)
        $localOrders = Order::where('phone', $request->phone)
            ->with(['items.product'])
            ->select('id', 'order_number', 'status', 'total_amount', 'created_at')
            ->latest()
            ->get()
            ->map(function ($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'status' => $order->status,
                    'amount' => $order->total_amount,
                    'date' => $order->created_at->format('d M Y, h:i A'),
                    'items' => $order->items->map(function ($item) {
                        return [
                            'product_name' => $item->product ? $item->product->name : 'Unknown Product',
                            'quantity' => $item->quantity,
                            'price' => $item->price,
                            'image' => $item->product && $item->product->thumbnail ? \Storage::url($item->product->thumbnail) : null,
                        ];
                    }),
                    'status_color' => match($order->status) {
                        'pending' => 'bg-yellow-100 text-yellow-800',
                        'confirmed' => 'bg-blue-100 text-blue-800',
                        'shipped' => 'bg-indigo-100 text-indigo-800',
                        'delivered' => 'bg-green-100 text-green-800',
                        'cancelled' => 'bg-red-100 text-red-800',
                        'returned' => 'bg-red-100 text-red-800',
                        default => 'bg-gray-100 text-gray-800',
                    }
                ];
            });

        // Always return 200, but with success flag, so frontend can handle message
        return response()->json([
            'success' => $result['success'],
            'data' => $result['data'] ?? null,
            'local_orders' => $localOrders,
            'percentage' => $result['percentage'] ?? 0,
            'message' => $result['message'] ?? null,
        ]);
    }
}
